<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class FeeStructureController extends Controller
{
    
    // ADD SINGLE RECORD
    public function addFeeStructure(Request $request)
    {
        $request->validate([
            'sProgram' => 'required|string',
            'sState' => 'required|string',
            'sInstituteUniqueId' => 'required|string',
            'sCollegeCode' => 'required|string',
            'sCourseName' => 'required|string',
            'sCollegeName' => 'required|string',
            'sDistrict' => 'required|string',
            'sFee' => 'required|array|min:1',
            'sFee.*.year' => 'required|string',
            'sFee.*.tuition_fee' => 'required|numeric',
            'sFee.*.development_fee' => 'required|numeric',
            'sFee.*.total_fee' => 'required|numeric',
            'sFee.*.fee_pdf' => 'required|url',
        ]);
    
        $data = $request->only([
            'sProgram', 'sState', 'sInstituteUniqueId', 'sCollegeCode',
            'sCourseName', 'sCollegeName', 'sDistrict'
        ]);
        $data['dtUpdatedDate'] = now();
        $data['iBisActive'] = 1;
    
        // ✅ Process fee structure
        $feeData = [];
        foreach ($request->input('sFee') as $fee) {
            $feeData[] = [
                'sYear' => $fee['year'], // Store the extracted year separately
                'tuition_fee' => $fee['tuition_fee'],
                'development_fee' => $fee['development_fee'],
                'total_fee' => $fee['total_fee'],
                'fee_pdf' => $fee['fee_pdf'],
            ];
        }
        $data['sFee'] = json_encode($feeData);
    
        try {
            $id = DB::table('tblFeeStructure')->insertGetId($data);
            return response()->json(['result' => '2', 'message' => 'Fee Structure Added', 'id' => $id], 201);
        } catch (\Exception $e) {
            return response()->json(['result' => '0', 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }
    



    // UPDATE RECORD
    public function updateFeeStructure(Request $request)
    {
        $request->validate([
            'sProgram' => 'sometimes|string',
            'sState' => 'sometimes|string',
            'sInstituteUniqueId' => 'sometimes|string',
            'sCollegeCode' => 'sometimes|string',
            'sCourseName' => 'sometimes|string',
            'sCollegeName' => 'sometimes|string',
            'sDistrict' => 'sometimes|string',
            'sFee' => 'sometimes|array',
        ]);
    
        $data = $request->only([
            'sProgram', 'sState', 'sInstituteUniqueId', 'sCollegeCode',
            'sCourseName', 'sCollegeName', 'sDistrict'
        ]);
        $data['dtUpdatedDate'] = now();
    
        if ($request->has('sFee')) {
            $feeData = [];
    
            foreach ($request->input('sFee') as $fee) {
                preg_match('/\b(20\d{2}-\d{2})\b/', $fee['tuition_fee'], $matches);
                $year = $matches[1] ?? null;
    
                $feeData[] = [
                    'sYear' => $year, 
                    'tuition_fee' => $fee['tuition_fee'], 
                    'development_fee' => $fee['development_fee'],
                    'total_fee' => $fee['total_fee'],
                    'fee_pdf' => $fee['fee_pdf'],
                ];
            }
    
            $data['sFee'] = json_encode($feeData);
        }
    
        $iId = $request->input('iId');
        try {
            $updated = DB::table('tblFeeStructure')->where('iId', $iId)->where('iBisActive', 1)->update($data);
            if ($updated) {
                return response()->json(['result' => '3', 'message' => 'Record updated'], 200);
            }
            return response()->json(['result' => '5', 'message' => 'No changes made or record inactive'], 200);
        } catch (\Exception $e) {
            return response()->json(['result' => '5', 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }
    


    // SOFT DELETE (SET iBisActive = 0)
    public function deleteFeeStructure($id)
    {
        try {
            $deleted = DB::table('tblFeeStructure')->where('iId', $id)->update(['iBisActive' => 0]);
            if ($deleted) {
                return response()->json(['result' => '1', 'message' => 'Record deactivated'], 200);
            }
            return response()->json(['result' => '4', 'message' => 'Record not found or already inactive'], 404);
        } catch (\Exception $e) {
            return response()->json(['result' => '5', 'message' => 'Error: ' . $e->getMessage()], 500);
        }
    }



     // GET ALL RECORDS (ONLY ACTIVE)
    public function getAllFeeStructure()
    {
        $records = DB::table('tblFeeStructure')->where('iBisActive', 1)->get();

        // Decode the JSON-encoded `sFee` field
        foreach ($records as $record) {
            $record->sFee = json_decode($record->sFee, true);
        }

        return response()->json(['result' => '1', 'data' => $records], 200);
    }



    // GET BY ID (ONLY IF ACTIVE)
    public function getByIdFeeStructure($id)
    {
        $record = DB::table('tblFeeStructure')->where('iId', $id)->where('iBisActive', 1)->first();

        if ($record) {
            // Decode the JSON-encoded `sFee` field
            $record->sFee = json_decode($record->sFee, true);

            return response()->json(['result' => '1', 'data' => $record], 200);
        }

        return response()->json(['result' => '0', 'message' => 'Record not found'], 404);
    }




    //  Add Multiple records
    public function addMultipleFeeStructure(Request $request)
    {
        try {
            // ✅ Validate input
            $request->validate([
                'records' => 'required|array|min:1',
                'records.*.sProgram' => 'required|string',
                'records.*.sState' => 'nullable|string',
                'records.*.sInstituteUniqueId' => 'required|string',
                'records.*.sCollegeCode' => 'required|string',
                'records.*.sCourseName' => 'required|string',
                'records.*.sCollegeName' => 'required|string',
                'records.*.sDistrict' => 'nullable|string',
                'records.*.sFee' => 'required|array|min:1',
            ]);
    
            // Extract request data
            $feeRecords = $request->input('records');
            $status = $request->input('status', 'append'); // Default to 'append'
            $sProgram = $request->input('sProgram', []); // Programs for conditional deletion
    
            \Log::info('Received status:', ['status' => $status]);
            \Log::info('Received sProgram:', ['sProgram' => $sProgram]);
    
            // ✅ Handle "addall" status - delete records based on program or clear the table
            if ($status === 'addall') {
                if (!empty($sProgram) && is_array($sProgram)) {
                    \Log::info('Programs to delete:', ['programs' => $sProgram]);
    
                    // Delete records for specific programs
                    foreach ($sProgram as $program) {
                        $deletedRows = DB::table('tblFeeStructure')->where('sProgram', $program)->delete();
                        \Log::info('Deleted rows for program', ['program' => $program, 'deleted' => $deletedRows]);
                    }
                } else {
                    // If no specific program provided, truncate the entire table
                    \Log::info('No programs provided, truncating the entire table.');
                    DB::table('tblFeeStructure')->truncate();
                }
            }
    
            // ✅ Prepare batch insert data
            $currentDate = now();
            $insertData = [];
    
            foreach ($feeRecords as $record) {
                $insertData[] = [
                    'sProgram' => $record['sProgram'],
                    'sState' => $record['sState'],
                    'sInstituteUniqueId' => $record['sInstituteUniqueId'],
                    'sCollegeCode' => $record['sCollegeCode'],
                    'sCourseName' => $record['sCourseName'],
                    'sCollegeName' => $record['sCollegeName'],
                    'sDistrict' => $record['sDistrict'],
                    'sFee' => json_encode($record['sFee'], JSON_UNESCAPED_UNICODE),
                    'iBisActive' => 1,
                    'dtUpdatedDate' => $currentDate,
                ];
            }
    
            // ✅ Batch insert in chunks (to handle large data efficiently)
            $batchSize = 1800;
            $chunks = array_chunk($insertData, $batchSize);
    
            foreach ($chunks as $chunk) {
                DB::table('tblFeeStructure')->insert($chunk);
            }
    
            \Log::info('Multiple Fee Structures Added successfully.');
    
            return response()->json(['result' => '2', 'message' => 'Multiple Fee Structures Added'], 201);
        } catch (\Exception $e) {
            // Log error for debugging
            \Log::error('Error in addMultipleFeeStructure:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json(['result' => '0', 'message' => 'An error occurred while adding fee structures.'], 500);
        }
    }
    
    
    
    
    
    
    
    public function getProgramsByFeeStructure(Request $request)
    {
    try {
        $query = DB::table('tblFeeStructure')
            ->select('sProgram', DB::raw('COUNT(*) AS count')) // Select distinct program and count
            ->whereNotNull('sProgram') // Ensure program is not null
            ->groupBy('sProgram'); // Group by program

        $result = $query->get(); // Execute query

        return response()->json([
            'success' => true,
            'data' => $result, // The distinct program data
            'message' => 'Programs retrieved successfully from fee structure.',
        ]);
    } catch (\Exception $e) {
        // Log the error if an exception occurs
        \Log::error('Error in getProgramsByFeeStructure:', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(), // Capture the stack trace for debugging
        ]);

        // Return a JSON response indicating failure
        return response()->json([
            'success' => false,
            'message' => 'An error occurred while retrieving programs.',
        ], 500);
    }
}



    

     // get Distinct state
     public function getDistinctStatefeeStruct(Request $request)
     {
         // Get the sProgram from the request
         $sProgram = $request->input('sProgram'); // Assuming the request contains 'sProgram'
 
         // Perform the query with sProgram filter and retrieve distinct sState
         $results = DB::table('tblFeeStructure')
             ->select(DB::raw('DISTINCT sState'), DB::raw('COUNT(sState) as count'))
             ->where('iBisActive', 1)
             ->whereNotNull('sState')
             ->where('sProgram', $sProgram) // Filter by sProgram
             ->groupBy('sState')
             ->get();
         
         // Return the results as JSON
         return response()->json($results);
     }





       // get Distinct Quota
    public function getDistinctDistrictForFee(Request $request)
    {
        // Retrieve 'sProgram' and 'sState' from the request
        $sProgram = $request->input('sProgram');
        $sState = $request->input('sState');
        
        // Build the query
        $query = DB::table('tblFeeStructure')
            ->select(DB::raw('DISTINCT sDistrict'), DB::raw('COUNT(sDistrict) as count'))
            ->where('iBisActive', 1)
            ->whereNotNull('sDistrict');
        
        // Apply additional filters if 'sProgram' and 'sState' are provided
        if ($sProgram) {
            $query->where('sProgram', $sProgram);  // Filter by sProgram
        }
        if ($sState) {
            $query->where('sState', $sState);  // Filter by sState
        }
        
        // Group by 'sQuota' to get the distinct values and count
        $results = $query->groupBy('sDistrict')->get();
        
        // Return the results as JSON
        return response()->json($results);
    }




    public function getDistinctCourseNamesforFee(Request $request)
    {
        try {
            // Retrieve input parameters from the request
            $sProgram = $request->input('sProgram');
            $sState = $request->input('sState');
            $sDistrict = $request->input('sDistrict'); // Fixed incorrect variable name
            
            // Start building the query
            $query = DB::table('tblFeeStructure')
                ->select('sCourseName', DB::raw('COUNT(*) as count')) // Fetch distinct course names and count
                ->where('iBisActive', 1)
                ->whereNotNull('sCourseName'); // Ensure course names are not null
            
            // Apply filters for the provided parameters
            if (!empty($sProgram)) {
                $query->where('sProgram', $sProgram); // Filter by program
            }
            if (!empty($sState)) {
                $query->where('sState', $sState); // Filter by state
            }
            if (!empty($sDistrict)) {
                $query->where('sDistrict', $sDistrict); // Filter by district
            }
    
            // Group by course name to get distinct values with count
            $query->groupBy('sCourseName');
    
            // Get the results
            $results = $query->get();
    
            // Return the results as a JSON response
            return response()->json([
                'success' => true,
                'data' => $results,
                'message' => 'Distinct courses retrieved successfully.',
            ]);
        } catch (\Exception $e) {
            // Log the error if an exception occurs
            \Log::error('Error in getDistinctCourseNamesforFee:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            // Return a JSON response indicating failure
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving distinct courses.',
            ], 500);
        }
    }





    // get Matrices according To Filter
    // public function getFeeStructureByFilter(Request $request)
    // {
    //     // Extract filters from "filters" key in request
    //     $filters = $request->input('filters', []);
    
    //     // Ensure we got an array of filters
    //     if (!is_array($filters) || empty($filters)) {
    //         return response()->json(['result' => '0', 'message' => 'Filters must be provided in the correct format.'], 400);
    //     }
    
    //     // Remove empty filters (trim spaces before checking)
    //     $filters = array_filter($filters, function ($value) {
    //         return !empty(trim($value)); // Trim whitespace and check for empty values
    //     });
    
    //     // Ensure at least one filter is provided
    //     if (empty($filters)) {
    //         return response()->json(['result' => '0', 'message' => 'At least one valid filter must be provided.'], 400);
    //     }
    
    //     try {
    //         $query = DB::table('tblFeeStructure');
    
    //         // Apply only non-empty filters
    //         foreach ($filters as $key => $value) {
    //             $query->where($key, $value);
    //         }
    
    //         $results = $query->get();
    
    //         // Decode JSON fields if needed
    //         $decodedResults = $results->map(function ($item) {
    //             $item->sFee = json_decode($item->sFee);
    //             return $item;
    //         });
    
    //         return response()->json(['result' => '2', 'data' => $decodedResults], 200);
    
    //     } catch (\Exception $e) {
    //         return response()->json(['result' => '0', 'message' => 'An error occurred: ' . $e->getMessage()], 500);
    //     }
    // }
 
 
     
    // public function getFeeStructureByFilter(Request $request)
    // {
    //     try {
    //         // Validate incoming request dynamically
    //         $validatedData = $request->validate([
    //             'query' => 'nullable|string',
    //             'per_page' => 'integer|nullable',
    //             'current_page' => 'integer|nullable',
    //             'sortField' => 'nullable|string',
    //             'sortOrder' => 'nullable|string',
    //             'filters' => 'nullable|array',
    //             'filters.program' => 'nullable|array',
    //             'filters.course' => 'nullable|array',
    //             'filters.state' => 'nullable|array',
    //             'filters.city' => 'nullable|array',
    //         ]);
    
    //         // Extract parameters
    //         $query = $validatedData['query'] ?? null;
    //         $perPage = $validatedData['per_page'] ?? 10;
    //         $currentPage = $validatedData['current_page'] ?? 1;
    //         $sortField = $validatedData['sortField'] ?? null;
    //         $sortOrder = $validatedData['sortOrder'] ?? 'asc';
    //         $filters = $validatedData['filters'] ?? [];
    
    //         // Build base query for tblFeeStructure without join
    //         $queryBuilder = DB::table('tblFeeStructure')
    //             ->select('tblFeeStructure.*');
    
    //         // Apply search query
    //         if (!empty($query)) {
    //             $searchableColumns = [
    //                 'tblFeeStructure.sProgram', 'tblFeeStructure.sCourseName', 'tblFeeStructure.sCollegeCode',
    //                 'tblFeeStructure.sBranchName', 'tblFeeStructure.sCollegeName'
    //             ];
    
    //             $queryBuilder->where(function ($q) use ($query, $searchableColumns) {
    //                 foreach ($searchableColumns as $column) {
    //                     $q->orWhere($column, 'LIKE', '%' . $query . '%');
    //                 }
    //             });
    //         }
    
    //         // Apply filters for program, course, state, and city
    //         if (!empty($filters['program'])) {
    //             $queryBuilder->whereIn('tblFeeStructure.sProgram', $filters['program']);
    //         }
    
    //         if (!empty($filters['course'])) {
    //             $queryBuilder->whereIn('tblFeeStructure.sCourseName', $filters['course']);
    //         }
    
    //         if (!empty($filters['state'])) {
    //             $queryBuilder->whereIn('tblFeeStructure.sState', $filters['state']);
    //         }
    
    //         if (!empty($filters['city'])) {
    //             $queryBuilder->whereIn('tblFeeStructure.sDistrict', $filters['city']);
    //         }
    
    //         // Apply sorting
    //         if ($sortField) {
    //             $queryBuilder->orderBy($sortField, $sortOrder);
    //         } else {
    //             $queryBuilder->orderBy('tblFeeStructure.sCollegeCode', 'asc');
    //         }
    
    //         // Pagination
    //         $total = $queryBuilder->count();
    //         $feeStructureList = $queryBuilder->offset(($currentPage - 1) * $perPage)->limit($perPage)->get();
    
    //         // Decode the sFee field for each item
    //         $feeStructureList = $feeStructureList->map(function ($item) {
    //             $item->sFee = json_decode($item->sFee, true);  // Decode the sFee field
    //             return $item;
    //         });
    
    //         $lastPage = (int) ceil($total / $perPage);
    
    //         // Return response
    //         return response()->json([
    //             'success' => true,
    //             'data' => $feeStructureList,
    //             'current_page' => $currentPage,
    //             'last_page' => $lastPage,
    //             'total' => $total,
    //             'per_page' => $perPage,
    //             'message' => 'Fee structure retrieved successfully.',
    //         ]);
    //     } catch (\Exception $e) {
    //         \Log::error('Error in getFeeStructureByFilter:', [
    //             'error' => $e->getMessage(),
    //             'trace' => $e->getTraceAsString(),
    //         ]);
    
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'An error occurred while retrieving fee structure data.',
    //         ], 500);
    //     }
    // }



    public function getFeeStructureByFilter(Request $request)
    {
        try {
            // Validate incoming request dynamically
            $validatedData = $request->validate([
                'query' => 'nullable|string',
                'per_page' => 'integer|nullable',
                'current_page' => 'integer|nullable',
                'sortField' => 'nullable|string',
                'sortOrder' => 'nullable|string',
                'filters' => 'nullable|array',
                'filters.program' => 'nullable|array',
                'filters.course' => 'nullable|array',
                'filters.state' => 'nullable|array',
                'filters.city' => 'nullable|array',
                'filters.fee' => 'nullable|array', // Accept multiple fee ranges
            ]);
    
            // Extract parameters
            $query = $validatedData['query'] ?? null;
            $perPage = $validatedData['per_page'] ?? 10;
            $currentPage = $validatedData['current_page'] ?? 1;
            $sortField = $validatedData['sortField'] ?? null;
            $sortOrder = $validatedData['sortOrder'] ?? 'asc';
            $filters = $validatedData['filters'] ?? [];
    
            // Build base query for tblFeeStructure
            $queryBuilder = DB::table('tblFeeStructure')->select('tblFeeStructure.*');
    
            // Apply search query
            if (!empty($query)) {
                $searchableColumns = [
                    'tblFeeStructure.sProgram', 'tblFeeStructure.sCourseName', 'tblFeeStructure.sCollegeCode', 'tblFeeStructure.sCollegeName'
                ];
                $queryBuilder->where(function ($q) use ($query, $searchableColumns) {
                    foreach ($searchableColumns as $column) {
                        $q->orWhere($column, 'LIKE', '%' . $query . '%');
                    }
                });
            }
    
            // Apply filters for program, course, state, and city
            if (!empty($filters['program'])) {
                $queryBuilder->whereIn('tblFeeStructure.sProgram', $filters['program']);
            }
            if (!empty($filters['course'])) {
                $queryBuilder->whereIn('tblFeeStructure.sCourseName', $filters['course']);
            }
            if (!empty($filters['state'])) {
                $queryBuilder->whereIn('tblFeeStructure.sState', $filters['state']);
            }
            if (!empty($filters['city'])) {
                $queryBuilder->whereIn('tblFeeStructure.sDistrict', $filters['city']);
            }
    
            // Apply Fee Range Filter
            if (!empty($filters['fee'])) {
                $queryBuilder->where(function ($query) use ($filters) {
                    foreach ($filters['fee'] as $feeRange) {
                        // Extract min and max fee from the given range
                        $feeParts = explode(" - ", str_replace(" above", "", $feeRange));
    
                        if (count($feeParts) === 2) {
                            $minFee = (int) $feeParts[0];
                            $maxFee = ($feeParts[1] === "above") ? 99999999 : (int) $feeParts[1]; // High value instead of PHP_INT_MAX
    
                            $query->orWhereRaw("
                                CAST(
                                    COALESCE(
                                        NULLIF(
                                            JSON_UNQUOTE(
                                                JSON_EXTRACT(tblFeeStructure.sFee, 
                                                    CONCAT('$[', JSON_LENGTH(tblFeeStructure.sFee) - 1, '].total_fee')
                                                )
                                            ), 'Yet to be declared'
                                        ),
                                        JSON_UNQUOTE(
                                            JSON_EXTRACT(tblFeeStructure.sFee, 
                                                CONCAT('$[', GREATEST(JSON_LENGTH(tblFeeStructure.sFee) - 2, 0), '].total_fee')
                                            )
                                        )
                                    ) AS UNSIGNED
                                ) BETWEEN ? AND ?
                            ", [$minFee, $maxFee]);
                        }
                    }
                });
            }
    
            // Apply sorting logic
            if ($sortField === 'sFee') {
                $queryBuilder->orderByRaw("
                    COALESCE(
                        NULLIF(
                            JSON_UNQUOTE(
                                JSON_EXTRACT(tblFeeStructure.sFee, 
                                    CONCAT('$[', JSON_LENGTH(tblFeeStructure.sFee) - 1, '].total_fee')
                                )
                            ), 'Yet to be declared'
                        ),
                        JSON_UNQUOTE(
                            JSON_EXTRACT(tblFeeStructure.sFee, 
                                CONCAT('$[', GREATEST(JSON_LENGTH(tblFeeStructure.sFee) - 2, 0), '].total_fee')
                            )
                        )
                    ) + 0 $sortOrder
                ");
            } else {
                $queryBuilder->orderBy($sortField ?? 'tblFeeStructure.sCollegeCode', $sortOrder);
            }
    
            // Pagination
            $total = $queryBuilder->count();
            $feeStructureList = $queryBuilder->offset(($currentPage - 1) * $perPage)->limit($perPage)->get();
    
            // Parse sFee JSON
            $feeStructureList = $feeStructureList->map(function ($item) {
                $item->sFee = json_decode($item->sFee, true); // Convert sFee JSON string to array
                return $item;
            });
    
            $lastPage = (int) ceil($total / $perPage);
    
            // Return response
            return response()->json([
                'success' => true,
                'data' => $feeStructureList,
                'current_page' => $currentPage,
                'last_page' => $lastPage,
                'total' => $total,
                'per_page' => $perPage,
                'message' => 'Fee structure retrieved successfully.',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getFeeStructureByFilter:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving fee structure data.',
            ], 500);
        }
    }





    // get distinct branch ny iCollegeCode
    public function getFeeByCollege(Request $request)
    {
        try {
            // Validate the input
            $validatedData = $request->validate([
                'iCollegeCode' => 'required|string',
            ]);
    
            // Extract inputs
            $collegecode = $validatedData['iCollegeCode'];
    
            // Query to fetch distinct branch names
            $result = DB::table('tblFeeStructure')
                ->select('sFee')
                ->where('sCollegeCode', $collegecode)
                ->get();
    
            // Handle empty results
            if ($result->isEmpty()) {
                return response()->json([
                    'success' => true,
                    'message' => 'No Fee structure found.',
                    'data' => [],
                ], 200);
            }
    
            // Decode JSON string to array
            $formattedResult = [];
            foreach ($result as $row) {
                $decodedFee = json_decode($row->sFee, true); // Decode JSON string
                if (is_array($decodedFee)) {
                    $formattedResult = array_merge($formattedResult, $decodedFee); // Merge into result array
                }
            }
    
            // Return response
            return response()->json([
                'success' => true,
                'message' => 'Fee Structure fetched successfully.',
                'data' => $formattedResult,
            ]);
    
        } catch (\Exception $e) {
            // Log error with request data
            \Log::error('Error in getFeeByCollege:', [
                'request' => $request->all(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while fetching fee structure.',
            ], 500);
        }
    }
    
    
    
     public function getFeeByCollegecode(Request $request)
    {
        try {
            // Validate the input
            $validatedData = $request->validate([
                'iCollegeCode' => 'required|string',
                'sProgram' => 'required|string'
            ]);
    
            // Extract inputs
            $collegecode = $validatedData['iCollegeCode'];
            $program = $validatedData['sProgram'];
    
            // Query to fetch distinct branch names
            $result = DB::table('tblFeeStructure')
                ->select('sFee')
                ->where('sCollegeCode', $collegecode)
                ->where('sProgram',$program)
                ->get();
    
            // Handle empty results
            if ($result->isEmpty()) {
                return response()->json([
                    'success' => true,
                    'message' => 'No Fee structure found.',
                    'data' => [],
                ], 200);
            }
    
            // Decode JSON string to array
            $formattedResult = [];
            foreach ($result as $row) {
                $decodedFee = json_decode($row->sFee, true); // Decode JSON string
                if (is_array($decodedFee)) {
                    $formattedResult = array_merge($formattedResult, $decodedFee); // Merge into result array
                }
            }
    
            // Return response
            return response()->json([
                'success' => true,
                'message' => 'Fee Structure fetched successfully.',
                'data' => $formattedResult,
            ]);
    
        } catch (\Exception $e) {
            // Log error with request data
            \Log::error('Error in getFeeByCollege:', [
                'request' => $request->all(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while fetching fee structure.',
            ], 500);
        }
    }
    
    
    public function getFeeRange(Request $request)
    {
        try {
            // Retrieve filter parameters from request
            $sProgram = $request->input('sProgram');
            $sState = $request->input('sState');
            $sDistrict = $request->input('sDistrict');
            $sCourseName = $request->input('sCourseName');
    
            // Start building the query
            $query = DB::table('tblFeeStructure')
                ->whereNotNull('sFee'); // Ensure sFee column has data
    
            // Apply filters
            if (!empty($sProgram)) {
                $query->where('sProgram', $sProgram);
            }
            if (!empty($sState)) {
                $query->where('sState', $sState);
            }
            if (!empty($sDistrict)) {
                $query->where('sDistrict', $sDistrict);
            }
            if (!empty($sCourseName)) {
                $query->where('sCourseName', $sCourseName);
            }
    
            // Fetch filtered fee structures
            $feeStructures = $query->get(['sFee']); // Get only sFee column
    
            $totalFees = [];
    
            // Loop through each record to extract the latest total_fee
            foreach ($feeStructures as $feeStructure) {
                $fees = json_decode($feeStructure->sFee, true); // Decode JSON
    
                if (!empty($fees) && is_array($fees)) {
                    // Get the last (latest) fee entry
                    $latestFee = end($fees);
    
                    // If the last fee entry is "Yet to be declared", fallback to previous entry
                    if (isset($latestFee['total_fee']) && strtolower($latestFee['total_fee']) !== 'yet to be declared') {
                        $totalFees[] = (int) $latestFee['total_fee'];
                    } elseif (count($fees) > 1) {
                        $prevFee = prev($fees);
                        if (isset($prevFee['total_fee']) && strtolower($prevFee['total_fee']) !== 'yet to be declared') {
                            $totalFees[] = (int) $prevFee['total_fee'];
                        }
                    }
                }
            }
    
            // If no fees found, return empty result
            if (empty($totalFees)) {
                return response()->json([
                    'success' => true,
                    'fee_ranges' => [],
                    'message' => 'No fee data available for the given filters.'
                ]);
            }
    
            // Determine the min and max fee values
            $minFee = min($totalFees);
            $maxFee = max($totalFees);
    
            // Define a dynamic range step based on the min and max fee
            $rangeStep = 10000;
            $ranges = [];
    
            // Generate fee ranges dynamically
            for ($start = floor($minFee / $rangeStep) * $rangeStep; $start <= $maxFee; $start += $rangeStep) {
                $end = $start + $rangeStep - 1;
                $ranges[] = [
                    "range" => "$start - $end",
                    "count" => 0
                ];
            }
            // Add a final range for values above the max range
            $ranges[] = [
                "range" => "$start - above",
                "count" => 0
            ];
    
            // Categorize fees into the dynamically generated ranges
            foreach ($totalFees as $fee) {
                foreach ($ranges as &$range) {
                    list($rangeStart, $rangeEnd) = explode(" - ", str_replace(" above", "", $range["range"]));
                    if ($fee >= $rangeStart && ($fee <= $rangeEnd || $rangeEnd === "above")) {
                        $range["count"]++;
                        break;
                    }
                }
            }
    
            return response()->json([
                'success' => true,
                'fee_ranges' => $ranges,
                'message' => 'Fee range distribution retrieved successfully.'
            ]);
        } catch (\Exception $e) {
            \Log::error('Error in getFeeRange:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while retrieving the fee range distribution.',
            ], 500);
        }
    }


    
}
