<?php

namespace App\Http\Controllers\App;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

use Illuminate\Support\Facades\Mail;
use App\Mail\SlotBookingConfirmationMail;
use Illuminate\Support\Facades\Http;


class SlotBookingController extends Controller
{

    
// Add  slot booking
//     public function addSlotBooking(Request $request)
// {
//     $request->validate([
//         'iSlotId' => 'required|integer',
//         'iStudentId' => 'required|integer',
//         'sReason' => 'nullable|string',
//     ]);

//     $data = $request->only(['iSlotId', 'iStudentId', 'sReason']);
//     $data['dtUpdatedDate'] = now();
//     $data['iBisActive'] = 1;

//     $id = DB::table('tblSlotBooking')->insertGetId($data);

//     // Fetch student email from tblRegistration
//     $student = DB::table('tblregistration')
//                 ->where('iId', $data['iStudentId'])
//                 ->first();

//     if ($student && $student->sEmailId) {
//         Mail::to($student->sEmailId)->send(new SlotBookingConfirmationMail($data));
//     }

//     return response()->json([
//         'result' => '2',
//         'message' => 'Slot added successfully',
//         'id' => $id
//     ], 201);
// }


   public function addSlotBooking(Request $request)
    {
        Log::info('Request received for adding slot booking', $request->all());
    
        $request->validate([
            'iSlotId' => 'required|integer',
            'iStudentId' => 'required|integer',
            'sReason' => 'nullable|string',
        ]);
        
        $data = $request->only(['iSlotId', 'iStudentId', 'sReason']);
        $data['dtUpdatedDate'] = now();
        $data['iBisActive'] = 1;
    
        // Log the data being processed
        Log::info('Booking data being processed', $data);
    
        // Get today's date in the correct format
        $todayDate = now()->format('Y-m-d');
        Log::info('Today\'s date: ' . $todayDate);
    
        // Fetch the slot details based on iSlotId (to ensure we're checking the correct slot)
        $slotDetails = DB::table('tblSlot')
            ->where('iId', $data['iSlotId'])
            ->first();
    
        // Check if the slot exists
        if (!$slotDetails) {
            return response()->json([
                'result' => '5',
                'message' => 'Invalid Slot ID',
            ], 400); // Return error if slot doesn't exist
        }
    
        // Check if the student has already booked a slot for this future date (dtSlotDate)
      $existingBooking = DB::table('tblSlotBooking')
        ->join('tblSlot', 'tblSlotBooking.iSlotId', '=', 'tblSlot.iId') // Join tblSlot on iSlotId
        ->where('tblSlotBooking.iStudentId', '=', $data['iStudentId']) // Match student ID
        ->where('tblSlotBooking.iBisActive', '=', 1) // Only active bookings
        ->whereDate('tblSlot.dtSlotDate', '=', $slotDetails->dtSlotDate) // Compare with dtSlotDate from tblSlot
        ->exists(); // Check if there is already a booking for that date
    
    
        // Log the query that is being executed and the result
        Log::info('Existing booking check for student ID ' . $data['iStudentId'], ['exists' => $existingBooking]);
    
        if ($existingBooking) {
            Log::warning('Student already has a booking for the selected slot date', ['studentId' => $data['iStudentId'], 'slotDate' => $slotDetails->dtSlotDate]);
            return response()->json([
                'result' => '5',
                'message' => 'Student already has a slot booked for this date.',
            ], 400); // Return error if the student already has a booking for the slot date
        }
    
        // Insert the new booking and get the ID
        $id = DB::table('tblSlotBooking')->insertGetId($data);
        Log::info('New slot booking created', ['bookingId' => $id]);
    
        // Fetch slot details by joining tblSlot
        $bookingDetails = DB::table('tblSlotBooking')
            ->join('tblSlot', 'tblSlotBooking.iSlotId', '=', 'tblSlot.iId')
            ->where('tblSlotBooking.iId', $id)
            ->select(
                'tblSlotBooking.iSlotId',
                'tblSlotBooking.sReason',
                'tblSlot.dtSlotDate',
                'tblSlot.sStartTime',
                'tblSlot.sEndTime'
            )
            ->first();
    
        // Log the fetched booking details
        Log::info('Booking details fetched', ['bookingDetails' => $bookingDetails]);
    
        // Fetch student email
        $student = DB::table('tblregistration')
            ->where('iId', $data['iStudentId'])
            ->first();
    
        // Log student details
        Log::info('Student details fetched', ['student' => $student]);
    
        if ($student && $student->sEmailId && $bookingDetails) {
            // Send the booking confirmation email
            Mail::to($student->sEmailId)->send(
                new SlotBookingConfirmationMail((array) $bookingDetails)
            );
            Log::info('Booking confirmation email sent to student', ['email' => $student->sEmailId]);
        }
        
        
        if ($student && $student->sStudentMobileNo && $bookingDetails) {
            $to = '91' . $student->sStudentMobileNo;
            $name = $student->sFirstName ?? 'Student';
            $date = $bookingDetails->dtSlotDate;
            $startTime = $bookingDetails->sStartTime;
            $endTime = $bookingDetails->sEndTime;
        
            // Static fields
            $location = 'https://maps.app.goo.gl/6gJrQwWoUJHEpirr5';
            $address = 'Dnyandeep, Adinath Housing Society, Behind DKTE YC Polytechnic, Sangli-Road, Sahakarnagar Ichalkarnji';
        
            try {
                $whatsappResponse = Http::post('https://mysoftway.com/whatsapp/webhook.php', [
                    'template' => 'book_slot_a2zadmission', // your actual WhatsApp template name
                    'to' => $to,
                    'name' => $name,
                    'date' => $date,
                    'startTime' => $startTime,
                    'endTime' => $endTime,
                    'location' => $location,
                    'address' => $address,
                ]);

        Log::info('Slot booking WhatsApp message sent', ['response' => $whatsappResponse->body()]);
    } catch (Exception $e) {
        Log::error('Failed to send WhatsApp slot booking message', ['error' => $e->getMessage()]);
    }
}

    
        // Return the success response with the booking ID
        Log::info('Returning response for booking creation', ['bookingId' => $id]);
    
        return response()->json([
            'result' => '2',
            'message' => 'Slot added successfully',
            'id' => $id
        ], 201);
    }



    // Update slot Booking
    public function updateSlotBooking(Request $request)
    {
        try {
            // Retrieve the input data
            $data = $request->only([
                'iSlotId', 'iStudentId', 'dtUpdatedDate'
            ]);
    
            // Retrieve the booking ID
            $iId = $request->input('iId');
            if (!$iId) {
                return response()->json(['message' => 'Booking ID is required'], 400);
            }
    
            // Set updated date
            $data['dtUpdatedDate'] = now();
    
            // Attempt to update the booking
            $updated = DB::table('tblSlotBooking')->where('iId', $iId)->update($data);
    
            if ($updated) {
                return response()->json(['result' => '3', 'message' => 'Slot updated successfully']);
            } else {
                Log::warning('Slot update failed', [
                    'iId' => $iId,
                    'data' => $data,
                ]);
                return response()->json(['message' => 'Slot failed to update'], 404);
            }
        } catch (\Exception $e) {
            // Log the error
            Log::error('An error occurred while updating the slot booking', [
                'error' => $e->getMessage(),
                'iId' => $request->input('iId'),
                'data' => $request->all(),
            ]);
    
            // Return a generic error response
            return response()->json(['message' => 'An unexpected error occurred'], 500);
        }
    }

    
    // Delete a record
    public function deleteSlotBooking($iId)
    {
        $deleted = DB::table('tblSlotBooking')->where('iId', $iId)->update(['iBisActive' => 0]);
        if ($deleted) {
            return response()->json(['result'=>'2','message' => 'Booked Slot deleted']);
        } else {
            return response()->json(['result'=>'2','message' => 'Booked Slot not found']);
        }
    }


    // get all slot Booking
    public function getAllSlotBooking()
    {
        $slotbooking = DB::table('tblSlotBooking')
            ->join('tblregistration', 'tblSlotBooking.iStudentId', '=', 'tblregistration.iId')
            ->join('tblSlot', 'tblSlotBooking.iSlotId', '=', 'tblSlot.iId') 
            ->select(
                'tblSlotBooking.*', 
                'tblregistration.sFirstName', 
                'tblSlot.dtSlotDate' ,
                'tblSlot.sStartTime',
                'tblSlot.sEndTime'
            )
            ->where('tblSlotBooking.iBisActive', 1)
            ->get();
    
        return response()->json(['result' => '1', 'data' => $slotbooking]);
    }
    


    // get slotbooking by studentid
    public function getSlotBookingByStudentId($iId)
    {
        $institute = DB::table('tblSlotBooking')
        ->where([
            ['iStudentId', '=', $iId],          
            ['iBisActive', '=', 1],    
        ])
        ->first();
            
        if ($institute) {
            return response()->json([
                'result' => '1', 
                'message' => 'Slot Booked found successfully',
                'data' => $institute, 
            ]);
        } else {
            return response()->json([
                'result' => '5',
                'message' => 'Slot Booked not found',
            ], 404);
        }
    }



    // Get slot by date
    public function getSlotBookedByDate(Request $request)
    {
        $date = $request->input('dtSlotDate');
    
        $slots = DB::table('tblSlotBooking')
            ->join('tblSlot', 'tblSlotBooking.iSlotid', '=', 'tblSlot.iId')
            ->join('tblregistration', 'tblSlotBooking.iStudentid', '=', 'tblregistration.iId')
            ->where([
                ['tblSlot.dtSlotDate', '=', $date],
                ['tblSlotBooking.iBisActive', '=', 1],
            ])
            ->select(
                'tblSlotBooking.*',
                'tblSlot.sStartTime',
                'tblSlot.sEndTime',
                'tblregistration.sFirstName',
                'tblregistration.sLastName',
                'tblregistration.sStudentMobileNo'
            )
            ->orderByRaw("STR_TO_DATE(tblSlot.sStartTime, '%r') ASC")  // '%r' = 12-hour time with seconds (hh:mm:ss AM/PM)
            ->orderByRaw("STR_TO_DATE(tblSlot.sEndTime, '%r') ASC")
            ->get();
    
        if ($slots->isNotEmpty()) {
            return response()->json([
                'result' => '1',
                'message' => 'Slots found successfully',
                'data' => $slots,
            ]);
        } else {
            return response()->json([
                'result' => '5',
                'message' => 'No slots found for the given date',
            ], 404);
        }
    }




    public function updateSlotBookingStatus(Request $request)
    {
        // 1️⃣ Validate input
        $request->validate([
            'iStudentId' => 'required|integer',
            'sStatus'    => 'required|string|max:50',
        ]);
    
        try {
            // 2️⃣ Update the status for active slot bookings of the student
            $updated = DB::table('tblSlotBooking')
                ->where('iStudentId', $request->iStudentId)
                ->where('iBisActive', 1)
                ->update([
                    'sStatus'       => $request->sStatus,
                    'dtUpdatedDate' => now()
                ]);
    
            if ($updated) {
                return response()->json([
                    'result'  => 1,
                    'message' => 'Slot Booking Status Updated Successfully'
                ]);
            }
    
            return response()->json([
                'result'  => 0,
                'message' => 'No Active Slot Booking Found for this Student'
            ]);
    
        } catch (\Exception $e) {
            return response()->json([
                'result'  => 0,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }




    
    public function getSlotBookingsByDateandStatus(Request $request)
    {
        // Optional filters
        $date   = $request->input('dtDate');    // format: YYYY-MM-DD
        $status = $request->input('sStatus');  // e.g., "Pending", "Completed", "Ongoing"
    
        $query = DB::table('tblSlotBooking as sb')
            ->leftJoin('tblregistration as s', 'sb.iStudentId', '=', 's.iId')
            ->where('sb.iBisActive', 1);
    
        // Filter by date if provided
        if ($date) {
            $query->whereDate('sb.dtUpdatedDate', $date);
        }
    
        // Filter by status if provided
        if ($status) {
            $query->where('sb.sStatus', $status);
        }
    
        $bookings = $query->select(
            'sb.*',
            's.sFirstName',
            's.sMiddleName',
            's.sLastName',
            's.sGender',
            's.sCity',
            's.sStudentMobileNo',
            's.sParentMobileNo',
            's.sEmailId'
        )->orderBy('sb.dtUpdatedDate', 'ASC')
        ->get();
    
        return response()->json([
            'result'  => 1,
            'message' => $bookings->count() ? 'Data Found' : 'No Data Found',
            'data'    => $bookings
        ]);
    }


    
    

}
