<?php

namespace App\Http\Controllers\App;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\ValidationException;
use Illuminate\Validation\Rule;
use App\Mail\SendPasswordMail1;
use App\Mail\SlotBookingConfirmationMail;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;


class appUserController extends Controller
{

    
// Get all records from tblregistration



    //Chnaged 24-01
    public function index(Request $request)
    {
        $fiscalYearId = $request->input('iFiscalYearId');

        if (!$fiscalYearId) {
            return response()->json([
                'result' => 0,
                'message' => 'Fiscal Year is required'
            ], 400);
        }
        // Subquery: payment totals per student
        $paymentSubquery = DB::table('tblPayment')
            ->select('iStudentid', DB::raw('SUM(fPaidAmount) as paid_amount'))
            ->groupBy('iStudentid');
    
        // Main query with category join (no groupBy issue)
        $registrations = DB::table('tblregistration as r')
            ->leftJoinSub($paymentSubquery, 'p', function ($join) {
                $join->on('r.iId', '=', 'p.iStudentid');
            })
            ->leftJoin('tblSocialCategoryOfStudents as c', 'r.iId', '=', 'c.iStudentid')
            ->select(
                'r.*',
                DB::raw('COALESCE(p.paid_amount, 0) as paid_amount'),
                'c.sCategory'
            )
            ->where('r.iBisActive', 1)
            ->where('r.iFiscalYearId', $fiscalYearId)
            
            ->where('r.sAppFlag', 1)
            ->get();
    
        if ($registrations->isEmpty()) {
            return response()->json([
                'result' => "6",
                'message' => 'No active registrations found',
            ], 404);
        }
    
        // Define prefix priority
        $prefixPriority = [
            'ME' => 0, 'EN' => 1, 'PH' => 2, 'AG' => 3,
            'AR' => 4, 'MB' => 5, 'MC' => 6, 'DS' => 7, 'ts' => 8,
        ];
    
        // Filter valid registrations
        $validRegistrations = $registrations->filter(function ($item) use ($prefixPriority) {
            return preg_match('/^([A-Za-z]+)(\d+)$/', $item->iRegistrationId, $matches)
                && isset($prefixPriority[$matches[1]]);
        });
    
        // Filter unknown registrations
        $unknownRegistrations = $registrations->filter(function ($item) {
            return empty($item->iRegistrationId) || preg_match('/Unknown/', $item->iRegistrationId);
        });
    
        // Sort valid registrations
        $validRegistrations = $validRegistrations->sortBy(function ($item) use ($prefixPriority) {
            preg_match('/^([A-Za-z]+)(\d+)$/', $item->iRegistrationId, $matches);
            $prefix = $matches[1] ?? 'Unknown';
            $number = isset($matches[2]) ? (int)$matches[2] : 0;
            $priority = $prefixPriority[$prefix] ?? 99;
    
            return [$priority, $number];
        });
    
        // Merge final list
        $registrations = $validRegistrations->merge($unknownRegistrations);
    
        // Final response
        return response()->json([
            'result' => "1",
            'message' => 'Active registrations retrieved successfully',
            'data' => $registrations,
        ], 200);
    }




    
    // Get a specific record by ID
    public function show($id)
    {
        // Fetch data from the database
        $registration = DB::table('tblregistration as r')
            ->leftJoin('tblAppUser as u', 'r.iId', '=', 'u.iStudentId')
            ->select(
                'r.*', // All registration columns
                'u.sName' // All AppUser columns
            )
            ->where('r.iId', $id)
            ->where('r.iBisActive', 1)
            ->first();
    
        // Ensure data exists
        if ($registration) {
            // Check and properly decode JSON fields
            foreach (['iCounsellingPackageId', 'iAppPackageId'] as $field) {
                if (!empty($registration->$field) && is_string($registration->$field)) {
                    $decoded = ($registration->$field);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $registration->$field = $decoded; // Convert string to array
                    } else {
                        Log::error("JSON Decode Error for {$field}", ['value' => $registration->$field]);
                    }
                }
            }
    
            // Return correctly formatted JSON
            return response()->json([
                'result' => "1",
                'data' => $registration
            ], 200, [], JSON_UNESCAPED_UNICODE);
        } 
        
        // Return error response if data not found
        return response()->json([
            'result' => "6",
            'message' => 'Registration not found'
        ], 404);
    }
    
    


    // Add a new registration

   public function store(Request $request)
    {
    try {
        $validatedData = $request->validate([
            'sFirstName' => 'required|string|max:255',
            'sMiddleName' => 'nullable|string|max:255',
            'sLastName' => 'nullable|string|max:255',
            'sGender' => 'nullable|string|in:Male,Female,Other',
            'sCity' => 'nullable|string|max:255',
            'sStudentMobileNo' => [
                'nullable',
                'string',
                'max:15',
               Rule::unique('tblregistration', 'sStudentMobileNo')
                ->where(function ($q) use ($request) {
                    $q->where('iBisActive', 1)
                      ->where('iFiscalYearId', $request->iFiscalYearId);
                }),
            ],
            'sParentMobileNo' => 'nullable|string|max:15',
            'sEmailId' => [
                'nullable',
                'email',
                'max:255',
                Rule::unique('tblregistration', 'sEmailId')->where(function ($query) {
                    return $query->where('iBisActive', 1);
                }),
            ],
            'iAmount' => 'nullable|integer|min:0',
            'iPartialAmount' => 'nullable|integer|min:0',
            'iConcession' => 'nullable|integer|min:0',
            'iTotalAmount' => 'nullable|integer|min:0',
            'iCounsellingPackageId' => 'nullable|array',
            'iCounsellingPackageId.*' => 'array',
            'iAppPackageId' => 'nullable|array',
            'sGroup' => 'nullable|string|max:10',
            'iAppPackageId.*' => 'array',
            'sActiveStatus' => 'nullable|string|in:Active,Inactive',
            'sFlag' => 'nullable|string',
            // 'iFiscalYearId'=>'nullable|integer',
            'sOtherDetails1' => 'nullable|array',
            'sOtherDetails1.*' => 'string',
            'sOtherDetails2' => 'nullable|string|max:255',
            'sOtherDetails3' => 'nullable|string|max:255',
            'sOtherDetails4' => 'nullable|string|max:255',
            'categoryForm' => 'nullable|array',
            'categoryForm.sCategory' => 'nullable|string',
            'categoryForm.sSpecialCategory' => 'nullable|string',
            'categoryForm.sOtherCategory' => 'nullable|string',
            'categoryForm.sMinorityStatus' => 'nullable|string',
            'categoryForm.sMinorityDetails' => 'nullable|string',
            'categoryForm.sAnnualIncome' => 'nullable|string',
            'categoryForm.sFathersOccupation' => 'nullable|string',
            'categoryForm.sMothersOccupation' => 'nullable|string',
            'categoryForm.sDisabilty' => 'nullable|string',
            'categoryForm.sDisabilityStatus' => 'nullable|string',
            'detailsForm' => 'nullable|array',
            'detailsForm.iFee' => 'nullable|numeric',
            'detailsForm.iConcession' => 'nullable|numeric',
            'detailsForm.iPayableAmount' => 'nullable|numeric',
            'detailsForm.details' => 'nullable|array',
            'detailsForm.details.*.sProgram' => 'nullable|string',
            'detailsForm.details.*.sState' => 'nullable|string',
            'detailsForm.details.*.sQuota' => 'nullable|string',
            'detailsForm.details.*.sCourseName' => 'nullable|string',
        ]);
    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json(['result' => 5, 'message' => 'Validation failed', 'errors' => $e->errors()], 422);
    }

    $generatedPassword = str_pad(random_int(100000, 999999), 6, '0', STR_PAD_LEFT);
    $hashedPassword = Hash::make($generatedPassword);

    $registrationData = [
        'sPassword' => $hashedPassword,
        'Salt' => '',
        'sFirstName' => $validatedData['sFirstName'],
        'sMiddleName' => $validatedData['sMiddleName'] ?? null,
        'sLastName' => $validatedData['sLastName'] ?? null,
        'sGender' => $validatedData['sGender'] ?? null,
        'sGroup' => $validatedData['sGroup'] ?? null,
        'sCity' => $validatedData['sCity'] ?? null,
        'sStudentMobileNo' => $validatedData['sStudentMobileNo'] ?? null,
        'sParentMobileNo' => $validatedData['sParentMobileNo'] ?? null,
        'sEmailId' => $validatedData['sEmailId'] ?? null,
        'iAmount' => $validatedData['iAmount'] ?? 0,
        'iPartialAmount' => $validatedData['iPartialAmount'] ?? 0,
        'iConcession' => $validatedData['iConcession'] ?? 0,
        'iTotalAmount' => $validatedData['iTotalAmount'] ?? 0,
        'iCounsellingPackageId' => json_encode($validatedData['iCounsellingPackageId'] ?? []),
        'iAppPackageId' => json_encode($validatedData['iAppPackageId'] ?? []),
        'sActiveStatus' => $validatedData['sActiveStatus'] ?? 'Inactive',
        'sFlag' => $validatedData['sFlag'] ?? '',
        'iFiscalYearId' => $validatedData['iFiscalYearId'] ?? null,
        'iBisActive' => 1,
        'dtUpdatedDate' => now(),
        'sOtherDetails1' => isset($validatedData['sOtherDetails1']) ? json_encode($validatedData['sOtherDetails1']) : json_encode([]),
        'sOtherDetails2' => $validatedData['sOtherDetails2'] ?? null,
        'sOtherDetails3' => $validatedData['sOtherDetails3'] ?? null,
        'sOtherDetails4' => $validatedData['sOtherDetails4'] ?? null,
    ];

    try {
        DB::beginTransaction();

        $registrationId = DB::table('tblregistration')->insertGetId($registrationData);

        DB::table('tblAppUser')->insert([
            'iStudentid' => $registrationId,
            'sName' => $validatedData['sFirstName'],
            'iBisActive' => 1,
            'dtUpdatedDate' => now(),
        ]);

        // Insert Social Category data if provided
        if (!empty($validatedData['categoryForm'])) {
            $category = $validatedData['categoryForm'];
            DB::table('tblSocialCategoryOfStudents')->insert([
                'iStudentId' => $registrationId,
                'sCategory' => $category['sCategory']?? null,
                'sSpecialCategory' => $category['sSpecialCategory'] ?? null,
                'sOtherCategory' => $category['sOtherCategory'] ?? null,
                'sMinorityStatus' => $category['sMinorityStatus'] ?? null,
                'sMinorityDetails' => $category['sMinorityDetails'] ?? null,
                'sAnnualIncome' => $category['sAnnualIncome'] ?? null,
                'sFathersOccupation' => $category['sFathersOccupation'] ?? null,
                'sMothersOccupation' => $category['sMothersOccupation'] ?? null,
                'sDisabilty' => $category['sDisabilty'] ?? null,
                'sDisabilityStatus' => $category['sDisabilityStatus'] ?? null,
                'dtUpdatedDate' => now(),
                'iBisActive' => 1
            ]);
        }

        // Insert Student Quota data if provided
       if (!empty($quota['details']) && is_array($quota['details'])) {
        foreach ($quota['details'] as $detail) {
            DB::table('tblStudentQuotaDetails')->insert([
                'iStudentId' => $registrationId,
                'sProgram' => $detail['sProgram'] ?? null,
                'sState' => $detail['sState'] ?? null,
                'sQuota' => $detail['sQuota'] ?? null,
                'sCourseName' => $detail['sCourseName'] ?? null,
                'dtUpdatedDate' => now(),
                'iBisActive' => 1,
            ]);
        }
    }


        DB::commit();

        // ✅ Send email if total amount is 0
        if (($validatedData['iTotalAmount'] ?? 0) == 0 && !empty($validatedData['sEmailId'])) {
            Log::info('Sending Email after registration with iTotalAmount = 0:', [
                'Username (Mobile)' => $validatedData['sStudentMobileNo'],
                'Parent Mobile' => $validatedData['sParentMobileNo'],
                'Password' => $generatedPassword
            ]);

            // Mail::to($validatedData['sEmailId'])->send(new SendPasswordMail1(
            //     $validatedData['sEmailId'],
            //     $generatedPassword,
            //     $validatedData['sStudentMobileNo'],
            //     $validatedData['sParentMobileNo']
            // ));
        }

        return response()->json([
            'result' => "2",
            'message' => 'All data inserted successfully',
            'id' => $registrationId,
            'password' => $generatedPassword
        ], 201);

    } catch (\Exception $e) {
        DB::rollBack();
        return response()->json([
            'result' => 0,
            'message' => 'Error occurred during registration',
            'error' => $e->getMessage()
        ], 500);
    }
}





    // Update a registration by iId

    public function update(Request $request)
    {
        try {
            // Validate request
            $validatedData = $request->validate([
                'iId' => 'required|numeric',
                'sFirstName' => 'nullable|string|max:255',
                'sMiddleName' => 'nullable|string|max:255',
                'sLastName' => 'nullable|string|max:255',
                'sGender' => 'nullable|string|in:Male,Female,Other',
                'sCity' => 'nullable|string|max:255',
                'sStudentMobileNo' => [
                    'nullable',
                    'string',
                    'max:15',
                    Rule::unique('tblregistration', 'sStudentMobileNo')
                        ->ignore($request->input('iId'), 'iId')
                        ->where(function ($query) {
                            $query->where('iBisActive', 1);
                        }),
                ],
                'sParentMobileNo' => 'nullable|string|max:15',
                'sEmailId' => [
                    'nullable',
                    'email',
                    'max:255',
                    Rule::unique('tblregistration', 'sEmailId')
                        ->ignore($request->input('iId'), 'iId')
                        ->where(function ($query) {
                            $query->where('iBisActive', 1);
                        }),
                ],
                'iAmount' => 'nullable|integer|min:0',
                'iPartialAmount' => 'nullable|integer|min:0',
                'iConcession' => 'nullable|integer|min:0',
                'iTotalAmount' => 'nullable|integer|min:0',
                'iCounsellingPackageId' => 'nullable|array',
                'iCounsellingPackageId.*' => 'array',
                'iAppPackageId' => 'nullable|array',
                'iAppPackageId.*' => 'array',
                'sActiveStatus' => 'nullable|string|in:Active,Inactive',
                'sFlag' => 'nullable|string',
                'sOtherDetails1' => 'nullable|array',
                'sOtherDetails1.*' => 'string',
                'sOtherDetails2' => 'nullable|string|max:255',
                'sOtherDetails3' => 'nullable|string|max:255',
                'sOtherDetails4' => 'nullable|string|max:255',
    
                // Optional Social Category
                'categoryForm' => 'nullable|array',
                'categoryForm.sCategory' => 'nullable|string',
                'categoryForm.sSpecialCategory' => 'nullable|string',
                'categoryForm.sOtherCategory' => 'nullable|string',
                'categoryForm.sMinorityStatus' => 'nullable|string',
                'categoryForm.sMinorityDetails' => 'nullable|string',
                'categoryForm.sAnnualIncome' => 'nullable|string',
                'categoryForm.sFathersOccupation' => 'nullable|string',
                'categoryForm.sMothersOccupation' => 'nullable|string',
                'categoryForm.sDisabilty' => 'nullable|string',
                'categoryForm.sDisabilityStatus' => 'nullable|string',
    
                // Optional Student Quota
                'detailsForm' => 'nullable|array',
                'detailsForm.iFee' => 'nullable|numeric',
                'detailsForm.iConcession' => 'nullable|numeric',
                'detailsForm.iPayableAmount' => 'nullable|numeric',
                'detailsForm.details' => 'nullable|array',
                'detailsForm.details.*.sProgram' => 'nullable|string',
                'detailsForm.details.*.sState' => 'nullable|string',
                'detailsForm.details.*.sQuota' => 'nullable|string',
                'detailsForm.details.*.sCourseName' => 'nullable|string',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation failed', $e->errors());
            return response()->json(['result' => 5, 'message' => 'Validation failed', 'errors' => $e->errors()], 422);
        }
    
        $studentId = $request->input('iId');
    
        $registrationData = [
            'sFirstName' => $validatedData['sFirstName'] ?? null,
            'sMiddleName' => $validatedData['sMiddleName'] ?? null,
            'sLastName' => $validatedData['sLastName'] ?? null,
            'sGender' => $validatedData['sGender'] ?? null,
            'sCity' => $validatedData['sCity'] ?? null,
            'sStudentMobileNo' => $validatedData['sStudentMobileNo'] ?? null,
            'sParentMobileNo' => $validatedData['sParentMobileNo'] ?? null,
            'sEmailId' => $validatedData['sEmailId'] ?? null,
            'iAmount' => $validatedData['iAmount'] ?? 0,
            'iPartialAmount' => $validatedData['iPartialAmount'] ?? 0,
            'iConcession' => $validatedData['iConcession'] ?? 0,
            'iTotalAmount' => $validatedData['iTotalAmount'] ?? 0,
            'iCounsellingPackageId' => isset($validatedData['iCounsellingPackageId']) ? json_encode($validatedData['iCounsellingPackageId']) : json_encode([]),
            'iAppPackageId' => isset($validatedData['iAppPackageId']) ? json_encode($validatedData['iAppPackageId']) : json_encode([]),
            'sActiveStatus' => $validatedData['sActiveStatus'] ?? 'Active',
            'sFlag' => $validatedData['sFlag'] ?? '',
            'iBisActive' => 1,
            'dtUpdatedDate' => now(),
            'sOtherDetails1' => isset($validatedData['sOtherDetails1']) ? json_encode($validatedData['sOtherDetails1']) : json_encode([]),
            'sOtherDetails2' => $validatedData['sOtherDetails2'] ?? null,
            'sOtherDetails3' => $validatedData['sOtherDetails3'] ?? null,
            'sOtherDetails4' => $validatedData['sOtherDetails4'] ?? null,
        ];
    
        try {
            DB::beginTransaction();
            Log::info('Starting update transaction for student ID: ' . $studentId);
    
            DB::table('tblregistration')
                ->where('iId', $studentId)
                ->update($registrationData);
            Log::info('Updated tblregistration.');
    
            DB::table('tblAppUser')
                ->where('iStudentid', $studentId)
                ->update([
                    'sName' => $validatedData['sFirstName'],
                    'iBisActive' => 1,
                    'dtUpdatedDate' => now(),
                ]);
            Log::info('Updated tblAppUser.');
    
            if (isset($validatedData['categoryForm'])) {
                $category = $validatedData['categoryForm'];
                Log::info('Replacing Social Category records.', $category);
    
                DB::table('tblSocialCategoryOfStudents')->where('iStudentId', $studentId)->delete();
                Log::info('Deleted old Social Category records.');
    
                DB::table('tblSocialCategoryOfStudents')->insert([
                    'iStudentId' => $studentId,
                    'sCategory' => $category['sCategory'],
                    'sSpecialCategory' => $category['sSpecialCategory'] ?? null,
                    'sOtherCategory' => $category['sOtherCategory'] ?? null,
                    'sMinorityStatus' => $category['sMinorityStatus'] ?? null,
                    'sMinorityDetails' => $category['sMinorityDetails'] ?? null,
                    'sAnnualIncome' => $category['sAnnualIncome'] ?? null,
                    'sFathersOccupation' => $category['sFathersOccupation'] ?? null,
                    'sMothersOccupation' => $category['sMothersOccupation'] ?? null,
                    'sDisabilty' => $category['sDisabilty'] ?? null,
                    'sDisabilityStatus' => $category['sDisabilityStatus'] ?? null,
                    'dtUpdatedDate' => now(),
                    'iBisActive' => 1,
                ]);
                Log::info('Inserted new Social Category record.');
            }
    
            if (isset($validatedData['detailsForm'])) {
                $quota = $validatedData['detailsForm'];
                Log::info('Replacing Student Quota.', $quota);
    
                DB::table('tblStudentQuota')
                    ->where('iStudentId', $studentId)
                    ->update([
                        'iFee' => $quota['iFee'] ?? 0,
                        'iConcession' => $quota['iConcession'] ?? 0,
                        'iPayableAmount' => $quota['iPayableAmount'] ?? 0,
                        'dtUpdatedDate' => now(),
                    ]);
                Log::info('Updated tblStudentQuota.');
    
                DB::table('tblStudentQuotaDetails')->where('iStudentId', $studentId)->delete();
                Log::info('Deleted old Quota details.');
    
                foreach ($quota['details'] ?? [] as $detail) {
                    DB::table('tblStudentQuotaDetails')->insert([
                        'iStudentId' => $studentId,
                        'sProgram' => $detail['sProgram'] ?? null,
                        'sState' => $detail['sState'] ?? null,
                        'sQuota' => $detail['sQuota'] ?? null,
                        'sCourseName' => $detail['sCourseName'] ?? null,
                        'dtUpdatedDate' => now(),
                        'iBisActive' => 1,
                    ]);
                }
                Log::info('Inserted new Quota details.');
            }
    
            DB::commit();
            Log::info('Transaction committed successfully.');
    
            return response()->json([
                'result' => "3",
                'message' => 'Data updated successfully',
                'id' => $studentId,
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error during update transaction: ' . $e->getMessage());
            return response()->json([
                'result' => 0,
                'message' => 'Error occurred during update',
                'error' => $e->getMessage(),
            ], 500);
        }
    }



    // Delete a registration by ID
    public function destroy($iId)
    {
        $registration = DB::table('tblregistration')->where('iId', $iId)->first();

        if (!$registration) {
            return response()->json(['result' => 0, 'message' => 'Registration not found'], 404);
        }

        $iStudentId = $registration->iId;

        DB::table('tblregistration')->where('iId', $iId)->update(['iBisActive' => 0]);
        DB::table('tblQualificationOfStudents')->where('iStudentId', $iStudentId)->update(['iBisActive' => 0]);
        DB::table('tblSocialCategoryOfStudents')->where('iStudentId', $iStudentId)->update(['iBisActive' => 0]);
        // DB::table('tblsubscription')->where('iStudentId', $iStudentId)->update(['iBisActive' => 0]);
        DB::table('tblStudentQuotaDetails')->where('iStudentId', $iStudentId)->update(['iBisActive' => 0]);

        return response()->json(['result' => "4", 'message' => 'All related records marked as inactive successfully'], 200);
    }




    // get filtered registrationa
    //  public function getFilteredRegistrations(Request $request)
    // {
    //     $program = $request->input('program');
    //     $status = $request->input('status');
    //     $paymentStatus = $request->input('payment'); // 'paid', 'partial', 'pending'
    
    //     // Base query with join and aggregation
    //     $query = DB::table('tblregistration as r')
    //         ->leftJoin('tblPayment as p', 'r.iId', '=', 'p.iStudentid')
    //         ->select(
    //             'r.*',
    //             DB::raw('COALESCE(SUM(p.fPaidAmount), 0) as paid_amount')
    //         )
    //         ->groupBy('r.iId');
    
    //     // Filter by program if present
    //     if ($program) {
    //         $query->where(function ($q) use ($program) {
    //             $q->whereRaw("JSON_CONTAINS(r.iCounsellingPackageId, JSON_OBJECT('sProgram', ?))", [$program])
    //               ->orWhereRaw("JSON_CONTAINS(r.iAppPackageId, JSON_OBJECT('sProgram', ?))", [$program]);
    //         });
    //     }
    
    //     // Filter by active status if present
    //     if ($status) {
    //         $query->where('r.sActiveStatus', $status);
    //     }
    
    //     // Filter by payment status if present
    //     if ($paymentStatus === 'paid') {
    //         $query->havingRaw('paid_amount = r.iTotalAmount');
    //     } elseif ($paymentStatus === 'partial') {
    //         $query->havingRaw('paid_amount > 0 AND paid_amount < r.iTotalAmount');
    //     } elseif ($paymentStatus === 'pending') {
    //         $query->havingRaw('paid_amount = 0');
    //     }
    
    //     $results = $query->get();
    
    //     // Add dynamic payment status label
    //     $results->transform(function ($item) {
    //         if ($item->paid_amount == 0) {
    //             $item->payment_status = 'Pending';
    //         } elseif ($item->paid_amount < $item->iTotalAmount) {
    //             $item->payment_status = 'Partial';
    //         } else {
    //             $item->payment_status = 'Paid';
    //         }
    //         return $item;
    //     });
    
    //     return response()->json([
    //         'result' => 1,
    //         'message' => 'Filtered records fetched successfully',
    //         'data' => $results
    //     ]);
    // }

    public function getFilteredRegistrations(Request $request)
    {
        $program = $request->input('program');
        $status = $request->input('status');
        $paymentStatus = $request->input('payment'); // 'paid', 'partial', 'pending'
        $fiscalYearId = $request->input('iFiscalYearId');
    
        if (!$fiscalYearId) {
            return response()->json([
                'result' => 0,
                'message' => 'Fiscal Year is required'
            ], 400);
        }
    
        // Subquery: calculate total paid per student
        $payments = DB::table('tblPayment')
            ->select('iStudentid', DB::raw('SUM(fPaidAmount) as paid_amount'))
            ->where('iFiscalYearId', $fiscalYearId)
            ->groupBy('iStudentid');
    
        // Main query: join registrations with payments
        $query = DB::table('tblregistration as r')
            ->leftJoinSub($payments, 'p', function ($join) {
                $join->on('r.iId', '=', 'p.iStudentid');
            })
            ->select(
                'r.*',
                DB::raw('COALESCE(p.paid_amount, 0) as paid_amount')
            )
            ->where('r.iBisActive', 1)        // only active students
            ->where('r.iFiscalYearId', $fiscalYearId)  // fiscal year filter
            ->where('r.sAppFlag', 1);         // only App-registered students
    
        // Optional: filter by program
        if ($program) {
            $query->where(function ($q) use ($program) {
                $q->whereRaw("JSON_CONTAINS(r.iCounsellingPackageId, JSON_OBJECT('sProgram', ?))", [$program])
                  ->orWhereRaw("JSON_CONTAINS(r.iAppPackageId, JSON_OBJECT('sProgram', ?))", [$program]);
            });
        }
    
        // Optional: filter by active status
        if ($status) {
            $query->where('r.sActiveStatus', $status);
        }
    
        // Optional: filter by payment status
        if ($paymentStatus === 'paid') {
            $query->whereRaw('COALESCE(p.paid_amount, 0) = r.iTotalAmount');
        } elseif ($paymentStatus === 'partial') {
            $query->whereRaw('COALESCE(p.paid_amount, 0) > 0 AND COALESCE(p.paid_amount, 0) < r.iTotalAmount');
        } elseif ($paymentStatus === 'pending') {
            $query->whereRaw('COALESCE(p.paid_amount, 0) = 0');
        }
    
        $results = $query->get();
    
        // Add dynamic payment status label
        $results->transform(function ($item) {
            if ($item->paid_amount == 0) {
                $item->payment_status = 'Pending';
            } elseif ($item->paid_amount < $item->iTotalAmount) {
                $item->payment_status = 'Partial';
            } else {
                $item->payment_status = 'Paid';
            }
            return $item;
        });
    
        return response()->json([
            'result' => 1,
            'message' => 'Filtered records fetched successfully',
            'data' => $results
        ]);
    }

   

    // login
    public function login(Request $request)
    {
        $validatedData = $request->validate([
            'mobileNo' => 'required|string|max:15',
            'sPassword' => 'required|string|min:4',
            'iDeviceId' => 'required|string',
        ]);
    
        $mobileNo = $validatedData['mobileNo'];
        $sPassword = $validatedData['sPassword'];
        $deviceId = $validatedData['iDeviceId'];
    
        Log::info("Login attempt for mobile: $mobileNo | Device ID: $deviceId | Password: $sPassword");
    
        // Step 1: Check if user exists
        $user = DB::table('tblregistration')
        ->where(function ($query) use ($mobileNo) {
            $query->where('sStudentMobileNo', $mobileNo)
                  ->orWhere('sParentMobileNo', $mobileNo);
        })
        ->where('iBisActive', 1)
        ->first();
    
    
        if (!$user) {
            Log::warning("Mobile number not found: $mobileNo");
            return response()->json([
                'result' => "10",
                'message' => "Mobile number not found."
            ]);
        }
    
        Log::info("User found. ID: $user->iId | sStudentMobileNo: $user->sStudentMobileNo | sParentMobileNo: $user->sParentMobileNo");
    
        // Step 2: Check account active status
        if ($user->iBisActive != 1 || $user->sActiveStatus !== 'Active') {
            Log::warning("Account inactive for user ID: $user->iId");
            return response()->json([
                'result' => "10",
                'message' => "Account is inactive."
            ]);
        }
    
        // Step 3: Check password
        if (!Hash::check($sPassword, $user->sPassword)) {
            Log::warning("Incorrect password for user ID: $user->iId. DB hash: $user->sPassword");
            return response()->json([
                'result' => "10",
                'message' => "Incorrect password."
            ]);
        }
    
        // Step 4: Determine login type
        $isStudentLogin = $user->sStudentMobileNo === $mobileNo;
        $isParentLogin = $user->sParentMobileNo === $mobileNo;
    
        Log::info("Login type: " . ($isStudentLogin ? 'Student' : 'Parent'));
    
        // Step 5: Check for device conflict
        if ($isStudentLogin && $user->sStudentDeviceId && $user->sStudentDeviceId !== $deviceId) {
            Log::warning("Student device ID mismatch. Existing: $user->sStudentDeviceId, Incoming: $deviceId");
            return response()->json([
                'result' => "11",
                'message' => "Student already logged in from another device."
            ]);
        }
    
        if ($isParentLogin && $user->sParentDeviceId && $user->sParentDeviceId !== $deviceId) {
            Log::warning("Parent device ID mismatch. Existing: $user->sParentDeviceId, Incoming: $deviceId");
            return response()->json([
                'result' => "11",
                'message' => "Parent already logged in from another device."
            ]);
        }
    
        // Step 6: Store device ID if missing
        if ($isStudentLogin && !$user->sStudentDeviceId) {
            Log::info("Updating student device ID for user ID: $user->iId");
            DB::table('tblregistration')->where('iId', $user->iId)->update([
                'sStudentDeviceId' => $deviceId,
            ]);
        }
    
        if ($isParentLogin && !$user->sParentDeviceId) {
            Log::info("Updating parent device ID for user ID: $user->iId");
            DB::table('tblregistration')->where('iId', $user->iId)->update([
                'sParentDeviceId' => $deviceId,
            ]);
        }
    
        // Get extra info
        $category = DB::table('tblSocialCategoryOfStudents')
            ->where('iStudentid', $user->iId)
            ->first();
    
        $loggedDeviceId = $isStudentLogin ? $user->sStudentDeviceId ?? $deviceId : $user->sParentDeviceId ?? $deviceId;
    
        Log::info("Login successful for user ID: $user->iId");
    
        return response()->json([
            'result' => "8",
            'message' => "Login successful.",
            'records' => [
                'sStudentMobileNo' => $user->sStudentMobileNo,
                'sParentMobileNo' => $user->sParentMobileNo,
                'iId' => $user->iId,
                'sFirstName' => $user->sFirstName,
                'iAmount' => $user->iAmount,
                'iCounsellingPackageId' => $user->iCounsellingPackageId,
                'iAppPackageId' => $user->iAppPackageId,
                'sFlag' => $user->sFlag,
                'sCategory' => $category->sCategory ?? null,
                'sActiveStatus' => $user->sActiveStatus,
                'sDeviceId' => $loggedDeviceId,
                'loginType' => $isStudentLogin ? 'student' : 'parent'
            ]
        ]);
    }



    //  Forget Password
   public function forgetPassword(Request $request)
   {
       $validatedData = $request->validate([
           'sStudentMobileNo' => 'required|string|max:15',
       ]);

       $sStudentMobileNo = $validatedData['sStudentMobileNo'];
       $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

       if ($user) {
           $otp = rand(100000, 999999);
           DB::table('tblregistration')
               ->where('sStudentMobileNo', $sStudentMobileNo)
               ->update(['sOTP' => $otp]);
               
                try {
                    $whatsappResponse = Http::post('https://mysoftway.com/whatsapp/webhook.php', [
                        'template' => 'forget_pass1',
                        'to' => '91' .  $sStudentMobileNo,
                        'code' => $otp
                       
                    ]);
    
    
                    Log::info('Password WhatsApp message sent:', [
                        'status' => $whatsappResponse->status(),
                        'response' => $whatsappResponse->body()
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to send password WhatsApp message: ' . $e->getMessage());
                }

           return response()->json(['result' => "8", 'otp' => $otp]);
       }

       return response()->json(['result' => "5", 'message' => "Mobile number not found."]);
   }




    //Reset Password
   public function resetPassword(Request $request)
   {
       $validatedData = $request->validate([
           'sStudentMobileNo' => 'required|string|max:15',
           'sOTP' => 'required|string',
           'sPassword' => 'required|string|min:4',
       ]);

       $sStudentMobileNo = $validatedData['sStudentMobileNo'];
       $sOTP = $validatedData['sOTP'];
       $newPassword = $validatedData['sPassword'];

       $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();

       if (!$user || $sOTP !== $user->sOTP) {
           return response()->json(['result' => "8", 'message' => "Invalid OTP or user not found."]);
       }

       $hashedPassword = Hash::make($newPassword);
       DB::table('tblregistration')
           ->where('sStudentMobileNo', $sStudentMobileNo)
           ->update(['sPassword' => $hashedPassword, 'sOTP' => null]);

       return response()->json(['result' => "9", 'message' => "Password reset successful."]);
   }




    // get details by mobileno
    public function getDetailsByMobileNo(Request $request)
    {
    // Validate that the mobile number is provided
    $validatedData = $request->validate([
        'mobileNo' => 'required|string|max:15',
    ]);

    try {
        // Retrieve the data from tblregistration based on the mobile number
        $registrationData = DB::table('tblregistration')
            ->where('sStudentMobileNo', $validatedData['mobileNo'])
            ->orWhere('sParentMobileNo', $validatedData['mobileNo']) // In case it's the parent mobile
            ->first(); // Using `first()` to get a single result

        // If no data is found, return an error response
        if (!$registrationData) {
            return response()->json([
                'result' => 0,
                'message' => 'No data found for the provided mobile number',
            ], 404);
        }

        // Return the found data as JSON response
        return response()->json([
            'result' => 1,
            'message' => 'Data retrieved successfully',
            'data' => $registrationData,
        ], 200);

    } catch (\Exception $e) {
        return response()->json([
            'result' => 0,
            'message' => 'Error occurred while retrieving data',
            'error' => $e->getMessage(),
        ], 500);
    }
}


    
    
    // registration

   public function registerAndBookSlot(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'sFirstName' => 'required|string|max:255',
                'sMiddleName' => 'nullable|string|max:255',
                'sLastName' => 'nullable|string|max:255',
                'sGender' => 'nullable|string|in:Male,Female,Other',
                'sCity' => 'nullable|string|max:255',
                'sStudentMobileNo' => 'nullable|string|max:15|unique:tblregistration,sStudentMobileNo',
                'sParentMobileNo' => 'nullable|string|max:15',
                'sEmailId' => 'nullable|email|max:255|unique:tblregistration,sEmailId',
                'iAmount' => 'nullable|integer|min:0',
                'iPartialAmount' => 'nullable|integer|min:0',
                'iConcession' => 'nullable|integer|min:0',
                'iTotalAmount' => 'nullable|integer|min:0',
                'iCounsellingPackageId' => 'nullable|array',
                'iCounsellingPackageId.*' => 'array',
                'iAppPackageId' => 'nullable|array',
                'iAppPackageId.*' => 'array',
                'sActiveStatus' => 'nullable|string|in:Active,Inactive',
                'sFlag' => 'nullable|string',
                'sOtherDetails1' => 'nullable|string|max:255',
                'sOtherDetails2' => 'nullable|string|max:255',
                'sOtherDetails3' => 'nullable|string|max:255',
                'sOtherDetails4' => 'nullable|string|max:255',
                'iSlotId' => 'required|integer',
                'sReason' => 'nullable|string',
            ]);

            $generatedPassword = str_pad(random_int(100000, 999999), 6, '0', STR_PAD_LEFT);
            $hashedPassword = Hash::make($generatedPassword);

            $registrationData = [
                'sPassword' => $hashedPassword,
                'Salt' => '',
                'sFirstName' => $validatedData['sFirstName'],
                'sMiddleName' => $validatedData['sMiddleName'] ?? null,
                'sLastName' => $validatedData['sLastName'] ?? null,
                'sGender' => $validatedData['sGender'] ?? null,
                'sCity' => $validatedData['sCity'] ?? null,
                'sStudentMobileNo' => $validatedData['sStudentMobileNo'] ?? null,
                'sParentMobileNo' => $validatedData['sParentMobileNo'] ?? null,
                'sEmailId' => $validatedData['sEmailId'] ?? null,
                'iAmount' => $validatedData['iAmount'] ?? 0,
                'iPartialAmount' => $validatedData['iPartialAmount'] ?? 0,
                'iConcession' => $validatedData['iConcession'] ?? 0,
                'iTotalAmount' => $validatedData['iTotalAmount'] ?? 0,
                'iCounsellingPackageId' => json_encode($validatedData['iCounsellingPackageId'] ?? []),
                'iAppPackageId' => json_encode($validatedData['iAppPackageId'] ?? []),
                'sActiveStatus' => $validatedData['sActiveStatus'] ?? 'Inactive',
                'sFlag' => $validatedData['sFlag'] ?? 'Pending',
                'iBisActive' => 1,
                'dtUpdatedDate' => now(),
                'sOtherDetails1' => $validatedData['sOtherDetails1'] ?? null,
                'sOtherDetails2' => $validatedData['sOtherDetails2'] ?? null,
                'sOtherDetails3' => $validatedData['sOtherDetails3'] ?? null,
                'sOtherDetails4' => $validatedData['sOtherDetails4'] ?? null,
            ];

            DB::beginTransaction();

            $registrationId = DB::table('tblregistration')->insertGetId($registrationData);

            DB::table('tblAppUser')->insert([
                'iStudentid' => $registrationId,
                'sName' => $validatedData['sFirstName'],
                'iBisActive' => 1,
                'dtUpdatedDate' => now(),
            ]);

            $slotBookingData = [
                'iSlotId' => $validatedData['iSlotId'],
                'iStudentId' => $registrationId,
                'sReason' => $validatedData['sReason'] ?? 'Counselling',

                'dtUpdatedDate' => now(),
                'iBisActive' => 1,
            ];

            DB::table('tblSlotBooking')->insert($slotBookingData);

            $slotDetails = DB::table('tblSlot')
                ->where('iId', $validatedData['iSlotId'])
                ->select('dtSlotDate', 'sStartTime', 'sEndTime')
                ->first();

            $student = DB::table('tblregistration')
                ->where('iId', $registrationId)
                ->select('sEmailId')
                ->first();

            if ($student && $student->sEmailId) {
                $emailData = [
                    'iSlotId' => $validatedData['iSlotId'],
                    'sReason' => $validatedData['sReason'] ?? 'Counselling',
                    'dtSlotDate' => $slotDetails->dtSlotDate ?? '',
                    'sStartTime' => $slotDetails->sStartTime ?? '',
                    'sEndTime' => $slotDetails->sEndTime ?? '',
                ];
                Mail::to($student->sEmailId)->send(new SlotBookingConfirmationMail($emailData));
            }
            
            
          if ($student && $slotDetails) {
                $to = '91' .$validatedData['sStudentMobileNo'];
                $name = $validatedData['sFirstName'] ?? 'Student';
                $date =  $slotDetails->dtSlotDate ?? '';
                $startTime = $slotDetails->sStartTime ?? '';
                $endTime = $slotDetails->sEndTime ?? '';
            
                // Static fields
                $location = 'https://maps.app.goo.gl/6gJrQwWoUJHEpirr5';
                $address = 'Dnyandeep, Adinath Housing Society, Behind DKTE YC Polytechnic, Sangli-Road, Sahakarnagar Ichalkarnji';
            
                try {
                    $whatsappResponse = Http::post('https://mysoftway.com/whatsapp/webhook.php', [
                        'template' => 'book_slot_a2zadmission', // Make sure this template exists and matches
                        'to' => $to,
                        'name' => $name,
                        'date' => $date,
                        'startTime' => $startTime,
                        'endTime' => $endTime,
                        'location' => $location,
                        'address' => $address,
                    ]);
            
                    Log::info('Slot booking WhatsApp message sent', ['response' => $whatsappResponse->body()]);
                } catch (Exception $e) {
                    Log::error('Failed to send WhatsApp slot booking message', ['error' => $e->getMessage()]);
                }
            }

            DB::commit();

            return response()->json([
                'result' => "2",
                'message' => 'Registration and slot booking successful',
                'id' => $registrationId,
                'password' => $generatedPassword,
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'result' => 0,
                'message' => 'Error occurred while registering and booking slot',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    // change device for student
    public function logoutStudent(Request $request)
    {
        $validated = $request->validate([
            'iId' => 'required|integer',
        ]);
    
        $user = DB::table('tblregistration')
            ->where('iId', $validated['iId'])
            ->first();
    
        if (!$user) {
            return response()->json([
                'result' => "0",
                'message' => "User not found.",
            ]);
        }
    
        if (is_null($user->sStudentDeviceId)) {
            return response()->json([
                'result' => "12",
                'message' => "Student already logged out.",
            ]);
        }
    
        $updated = DB::table('tblregistration')
            ->where('iId', $validated['iId'])
            ->update(['sStudentDeviceId' => null]);
    
        if ($updated) {
            return response()->json([
                'result' => "3",
                'message' => "Student logged out successfully.",
            ]);
        } else {
            return response()->json([
                'result' => "0",
                'message' => "Student logout failed due to an error.",
            ]);
        }
    }





    // change device for parent
    public function logoutParent(Request $request)
    {
        $validated = $request->validate([
            'iId' => 'required|integer',
        ]);
    
        $user = DB::table('tblregistration')
            ->where('iId', $validated['iId'])
            ->first();
    
        if (!$user) {
            return response()->json([
                'result' => "0",
                'message' => "User not found.",
            ]);
        }
    
        if (is_null($user->sParentDeviceId)) {
            return response()->json([
                'result' => "12",
                'message' => "Parent already logged out.",
            ]);
        }
    
        $updated = DB::table('tblregistration')
            ->where('iId', $validated['iId'])
            ->update(['sParentDeviceId' => null]);
    
        if ($updated) {
            return response()->json([
                'result' => "3",
                'message' => "Parent logged out successfully.",
            ]);
        } else {
            return response()->json([
                'result' => "0",
                'message' => "Parent logout failed due to an error.",
            ]);
        }
    }


    public function deActivateAccount(Request $request)
    {
         $validated = $request->validate([
           'iId' => 'required|integer',
        ]);
        
        $user = DB::table('tblregistration')
            ->where('iId', $validated['iId'])
            ->first();
        
        if (!$user) {
            return response()->json([
                'result' => "0",
                'message' => "User not found.",
            ]);
        }
        
        $parentId = $user->sParentDeviceId;
        $studentId = $user->sStudentDeviceId;
        
        // Always prepare update with sActiveStatus = 'InActive'
        $updateData = [
            'sActiveStatus' => 'Inactive',
        ];
        
        // If not null, set them to null
        if (!is_null($parentId)) {
            $updateData['sParentDeviceId'] = null;
        }
        if (!is_null($studentId)) {
            $updateData['sStudentDeviceId'] = null;
        }
        
        // Perform the update regardless of device ID state
        $updated = DB::table('tblregistration')
            ->where('iId', $validated['iId'])
            ->update($updateData);
        
        if (is_null($parentId) && is_null($studentId)) {
            return response()->json([
                'result' => "12",
                'message' => "Parent and student already logged out. Status set to inactive.",
            ]);
        }
        
        if ($updated) {
            return response()->json([
                'result' => "3",
                'message' => "Logout successful. Status set to inactive.",
            ]);
        } else {
            return response()->json([
                'result' => "0",
                'message' => "Logout failed due to an error.",
            ]);
        }


    }
    
    


   
   
    //   validate device
   public function validateDevice(Request $request)
   {
    // Validate incoming request data
    $validatedData = $request->validate([
        'iId' => 'required|integer',         // User ID (Student or Parent)
        'iDeviceId' => 'required|string',    // Device ID to be validated
    ]);

    // Retrieve user data based on user ID
    $user = DB::table('tblregistration')->where('iId', $validatedData['iId'])->first();

    // Check if user exists
    if (!$user) {
        return response()->json([
            'result' => '0',
            'message' => 'User not found.',
        ]);
    }

    // Validate device ID based on user type
    if ($user->sStudentDeviceId && $user->sStudentDeviceId === $validatedData['iDeviceId']) {
        return response()->json([
            'result' => '1',
            'message' => 'Device validated successfully for student.',
        ]);
    }

    if ($user->sParentDeviceId && $user->sParentDeviceId === $validatedData['iDeviceId']) {
        return response()->json([
            'result' => '1',
            'message' => 'Device validated successfully for parent.',
        ]);
    }

    // If device ID doesn't match for either student or parent
    return response()->json([
        'result' => '0',
        'message' => 'Logged in from another device.',
    ]);
}
 
 
 
 
    // get data by specific date
    public function getRegistrationsByDate(Request $request)
    {
        // Validate the date parameter
        $request->validate([
            'date' => 'required|date_format:Y-m-d',
        ]);
    
        // Get the date from the request
        $date = $request->input('date');
    
        // Query to fetch registrations for the given date and iBisActive = 1 (active registrations)
        $registrations = DB::table('tblregistration as r')
            ->leftJoin('tblPayment as p', 'r.iId', '=', 'p.iStudentid')
            ->select(
                'r.*',
                DB::raw('COALESCE(SUM(p.fPaidAmount), 0) as paid_amount')
            )
            ->where('r.iBisActive', 1)  // Only active registrations (iBisActive = 1)
            ->whereDate('r.dtUpdatedDate', '=', $date) // Filter by the specified date
            ->groupBy('r.iId')
            ->get();
    
        // Check if any registrations were found
        if ($registrations->isEmpty()) {
            return response()->json([
                'result' => "6",
                'message' => 'No active registrations found for the specified date',
            ], 404);
        }
    
        return response()->json([
            'result' => "1",
            'message' => 'Active registrations retrieved successfully',
            'data' => $registrations,
        ], 200);
    }



 
     public function resetPasswordforadmin(Request $request)
     {
           $validatedData = $request->validate([
               'studentno' => 'required|string|max:15',
               'password' => 'required|string|min:4',
           ]);
    
           $sStudentMobileNo = $validatedData['studentno'];
           $newPassword = $validatedData['password'];
    
           $user = DB::table('tblregistration')->where('sStudentMobileNo', $sStudentMobileNo)->where('iBisActive', 1)->first();
    
    
           $hashedPassword = Hash::make($newPassword);
           DB::table('tblregistration')
               ->where('sStudentMobileNo', $sStudentMobileNo)
               ->update(['sPassword' => $hashedPassword, 'sOTP' => null]);
    
           return response()->json(['result' => "9", 'message' => "Password reset successful."]);
     }
   
   
   
   
   //NEW
  
   
   /**
     * Deactivate multiple registrations
     */
    public function deactivate(Request $request)
    {
        // Validate input
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'integer',
        ]);

        $ids = $request->input('ids');

        try {
            $updated = DB::table('tblregistration')
                ->whereIn('iId', $ids)
                ->update([
                    'sActiveStatus' => 'Inactive',
                    'dtUpdatedDate' => now(),
                ]);

            return response()->json([
                'result' => 1,
                'message' => "$updated record(s) deactivated successfully.",
                'deactivated_ids' => $ids,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'result' => 0,
                'message' => 'Failed to deactivate records: ' . $e->getMessage(),
            ], 500);
        }
    }




    // Get distinct Grups
    public function getDistinctGroups()
    {
        try {
            $groups = DB::table('tblregistration')
                ->where('iBisActive', 1)
                ->selectRaw("DISTINCT COALESCE(sGroup, 'No Group') AS sGroup")
                ->pluck('sGroup'); // Returns a simple array of group names
    
            return response()->json([
                'result' => 1,
                'message' => 'Groups fetched successfully.',
                'groups' => $groups,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'result' => 0,
                'message' => 'Failed to fetch groups: ' . $e->getMessage(),
            ], 500);
        }
    }
    
    
    
   
    public function getStudentsByGroup(Request $request)
    {
        $request->validate([
            'sGroup' => 'required|string',
            'iFiscalYearId' => 'required|integer',
        ]);
    
        try {
            $group = $request->input('sGroup');
            $fiscalYearId = $request->input('iFiscalYearId');
    
            $query = DB::table('tblregistration')
                ->select(
                    'iId',
                    'iRegistrationId',
                    'sFirstName',
                    'sMiddleName',
                    'sLastName',
                    'sGender',
                    'sCity',
                    'sStudentMobileNo',
                    'sParentMobileNo',
                    'sEmailId',
                    'iAmount',
                    'iPartialAmount',
                    'iConcession',
                    'iTotalAmount',
                    'sGroup',
                    'sActiveStatus',
                    'sFlag',
                    'iCounsellingPackageId',
                    'iAppPackageId'
                )
                ->where('sActiveStatus', 'Active')
                ->whereNotNull('iRegistrationId')
                ->where('iFiscalYearId', $fiscalYearId);
    
            // 🔹 Group conditions
            if (strtolower($group) === 'all') {
                // no extra filter
            } elseif (strtolower($group) === 'no group') {
                $query->whereNull('sGroup');
            } else {
                $query->where('sGroup', $group);
            }
    
            $students = $query->get();
    
            if ($students->isEmpty()) {
                return response()->json([
                    'result' => 6,
                    'message' => 'No students found for selected group and fiscal year',
                    'data' => []
                ], 404);
            }
    
            return response()->json([
                'result' => 1,
                'message' => 'Students fetched successfully',
                'data' => $students
            ], 200);
    
        } catch (\Exception $e) {
            return response()->json([
                'result' => 0,
                'message' => 'Failed to fetch students',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    //send OTP function
    public function sendOtp(Request $request)
    {
        $request->validate([
            'mobileNo'  => 'required|string|max:15',
            'iDeviceId' => 'required|string|max:100',
        ]);
    
        $mobileNo = $request->mobileNo;
        $deviceId = $request->iDeviceId;
    
        $user = DB::table('tblregistration')
            ->where(function ($query) use ($mobileNo) {
                $query->where('sStudentMobileNo', $mobileNo)
                      ->orWhere('sParentMobileNo', $mobileNo);
            })
            ->where('iBisActive', 1)
            ->first();
    
        if (!$user || $user->sActiveStatus !== 'Active') {
            return response()->json([
                'result'  => "10",
                'message' => "Mobile number not found."
            ]);
        }
    
        // 🔍 Identify role
        $isStudent = ($user->sStudentMobileNo === $mobileNo);
        $isParent  = ($user->sParentMobileNo === $mobileNo);
    
        // 🎯 Select correct device column
        $deviceColumn = $isStudent ? 'sStudentDeviceId' : 'sParentDeviceId';
        $storedDevice = $user->$deviceColumn;
    
        // ✅ First-time login → save device ID
        if (empty($storedDevice)) {
            DB::table('tblregistration')
                ->where('iId', $user->iId)
                ->update([
                    $deviceColumn => $deviceId
                ]);
        }
        // ❌ Device mismatch
        elseif ($storedDevice !== $deviceId) {
            return response()->json([
                'result'  => "11",
                'message' => "Already logged in on another device."
            ]);
        }
    
        // ✅ Send OTP
        $otp = rand(1000, 9999);
    
        DB::table('tblregistration')
            ->where('iId', $user->iId)
            ->update([
                'sOTP' => $otp,
                'otp_created_at' => now()
            ]);
    
        $sendToMobile = $mobileNo;
    
        Log::info("OTP sent to $sendToMobile | OTP: $otp");
    
        try {
            $whatsappResponse = Http::post('https://mysoftway.com/whatsapp/webhook.php', [
                'template' => 'forget_pass1',
                'to'       => '91' . $sendToMobile,
                'code'     => $otp
            ]);
    
            Log::info("WhatsApp OTP sent to {$sendToMobile}", [
                'response' => $whatsappResponse->body()
            ]);
    
        } catch (\Exception $e) {
            Log::error("WhatsApp OTP sending failed for {$sendToMobile}", [
                'error' => $e->getMessage()
            ]);
        }
    
        return response()->json([
            'result'  => "8",
            'message' => "OTP sent successfully."
        ]);
    }





    //Verify OTP
    public function verifyOtp(Request $request)
    {
        $request->validate([
            'mobileNo' => 'required|string|max:15',
            'sOTP'     => 'required|string|size:4',
        ]);
    
        $mobileNo = $request->mobileNo;
        $otp      = $request->sOTP;
    
        // Get user by OTP + mobile (student or parent)
        $user = DB::table('tblregistration')
            ->where(function ($query) use ($mobileNo) {
                $query->where('sStudentMobileNo', $mobileNo)
                      ->orWhere('sParentMobileNo', $mobileNo);
            })
            ->where('sOTP', $otp)
            ->where('iBisActive', 1)
            ->where('sActiveStatus', 'Active')
            ->first();
    
        if (!$user) {
            return response()->json([
                'result'  => "10",
                'message' => "Invalid OTP."
            ]);
        }
        
        if ($user->sActiveStatus !== 'Active') {
            return response()->json([
                'result' => 0,
                'message' => 'Your account is inactive. Please contact admin.'
            ], 403);
        }
            
        // Check OTP expiry (10 minutes)
        if ($user->otp_created_at && now()->diffInMinutes($user->otp_created_at) > 10) {
            return response()->json([
                'result'  => "10",
                'message' => "OTP expired."
            ]);
        }
    
        // Clear OTP after successful verification
        DB::table('tblregistration')
            ->where('iId', $user->iId)
            ->update(['sOTP' => null]);
    
        // Return **all fields except sensitive ones**
        $userDetails = [
            'iId'                  => $user->iId,
            'iRegistrationId'      => $user->iRegistrationId,
            'sFirstName'           => $user->sFirstName,
            'sMiddleName'          => $user->sMiddleName,
            'sLastName'            => $user->sLastName,
            'sGender'              => $user->sGender,
            'sCity'                => $user->sCity,
            'sStudentMobileNo'     => $user->sStudentMobileNo,
            'sParentMobileNo'      => $user->sParentMobileNo,
            'sEmailId'             => $user->sEmailId,
            'iAmount'              => $user->iAmount,
            'iPartialAmount'       => $user->iPartialAmount,
            'iConcession'          => $user->iConcession,
            'iTotalAmount'         => $user->iTotalAmount,
            'iCounsellingPackageId' => $user->iCounsellingPackageId,
            'iAppPackageId'        => $user->iAppPackageId,
            'sActiveStatus'        => $user->sActiveStatus,
            'sFlag'                => $user->sFlag,
            'sParentDeviceId'      => $user->sParentDeviceId,
            'sStudentDeviceId'     => $user->sStudentDeviceId,
            'iBisActive'           => $user->iBisActive,
            'dtUpdatedDate'        => $user->dtUpdatedDate,
            'iUpdateby'            => $user->iUpdateby,
            'sOtherDetails1'       => $user->sOtherDetails1,
            'sGroup'               => $user->sGroup,
            'iFiscalYearId'        => $user->iFiscalYearId,
            'otp_created_at'       => $user->otp_created_at,
            'sAppFlag'             => $user->sAppFlag ?? null
        ];
    
        return response()->json([
            'result'  => "8",
            'message' => "OTP verified successfully.",
            'records' => $userDetails
        ]);
    }


    
    //REsend OTP
    public function resendOtp(Request $request)
    {
        $request->validate([
            'mobileNo' => 'required|string|max:15',
        ]);
    
        $mobileNo = $request->mobileNo;
    
        // 🔍 Find user by student OR parent mobile
        $user = DB::table('tblregistration')
            ->where(function ($query) use ($mobileNo) {
                $query->where('sStudentMobileNo', $mobileNo)
                      ->orWhere('sParentMobileNo', $mobileNo);
            })
            ->where('iBisActive', 1)
            ->first();
    
        if (!$user || $user->sActiveStatus !== 'Active') {
            return response()->json([
                'result'  => "10",
                'message' => "Mobile number not found."
            ]);
        }
    
        // 🔁 Generate new OTP (4 digit)
        $otp = rand(1000, 9999);
    
        DB::table('tblregistration')
            ->where('iId', $user->iId)
            ->update([
                'sOTP' => $otp,
                'otp_created_at' => now()
            ]);
    
        Log::info("OTP resent to {$mobileNo} | OTP: {$otp}");
    
        try {
            $whatsappResponse = Http::post('https://mysoftway.com/whatsapp/webhook.php', [
                'template' => 'forget_pass1',
                'to'       => '91' . $mobileNo,
                'code'     => $otp
            ]);
    
            Log::info("WhatsApp OTP resent to {$mobileNo}", [
                'response' => $whatsappResponse->body()
            ]);
    
        } catch (\Exception $e) {
            Log::error("WhatsApp OTP resend failed for {$mobileNo}", [
                'error' => $e->getMessage()
            ]);
        }
    
        return response()->json([
            'result'  => "8",
            'message' => "OTP resent successfully."
        ]);
    }






    public function verify_mobile(Request $request)
    {
        try {
            $request->validate([
                'sStudentMobileNo' => 'required|string|max:20'
            ]);
    
            $today = Carbon::today()->toDateString();
    
            $student = DB::table('tblregistration as r')
                ->leftJoin('tblSlotBooking as sb', function ($join) {
                    $join->on('sb.iStudentId', '=', 'r.iId')
                         ->where('sb.iBisActive', 1);
                })
                ->leftJoin('tblSlot as s', function ($join) use ($today) {
                    $join->on('s.iId', '=', 'sb.iSlotId')
                         ->whereDate('s.dtSlotDate', $today)
                         ->where('s.iBisActive', 1);
                })
                ->where('r.sStudentMobileNo', $request->sStudentMobileNo)
                ->where('r.iBisActive', 1)
                ->select(
                    'r.*',
                    'sb.iId as booking_id',
                    'sb.sStatus as booking_status',
                    's.dtSlotDate',
                    's.sStartTime',
                    's.sEndTime'
                )
                ->first();
    
            if ($student) {
                return response()->json([
                    'result' => 1,
                    'message' => 'Student Found',
                    'data' => $student
                ]);
            }
    
            return response()->json([
                'result' => 2,
                'message' => 'Student Not Found'
            ]);
    
        } catch (\Exception $e) {
            return response()->json([
                'result' => 5,
                'message' => 'Error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    //Store BAsic details by and also for App Register
    public function storeStudentsBasicDetailsAll(Request $request)
    {
        try {
            $validated = $request->validate([
                'sFirstName' => 'required|string|max:50',
                'sMiddleName' => 'nullable|string|max:50',
                'sLastName' => 'required|string|max:50',
                'sGender' => 'nullable|string|max:20',
                'sCity' => 'nullable|string|max:50',
                'sStudentMobileNo' => 'required|string|max:20',
                'sParentMobileNo' => 'nullable|string|max:20',
                'sEmailId' => 'nullable|email|max:50',
                'iFiscalYearId' => 'required|integer'
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'result' => 5,
                'message' => 'Validation Failed',
                'errors' => $e->errors()
            ], 422);
        }
    
        try {
            DB::beginTransaction();
    
            // 1️⃣ Insert into tblregistration
            $studentId = DB::table('tblregistration')->insertGetId([
                'sFirstName' => $validated['sFirstName'],
                'sMiddleName' => $validated['sMiddleName'] ?? null,
                'sLastName' => $validated['sLastName'],
                'sGender' => $validated['sGender'] ?? null,
                'sCity' => $validated['sCity'] ?? null,
                'sStudentMobileNo' => $validated['sStudentMobileNo'],
                'sParentMobileNo' => $validated['sParentMobileNo'] ?? null,
                'sEmailId' => $validated['sEmailId'] ?? null,
                'iFiscalYearId' => $validated['iFiscalYearId'] ?? null,
                'iBisActive' => 1,
                'dtUpdatedDate' => now()
            ]);
    
            // 2️⃣ Auto insert into tblAppUser
            DB::table('tblAppUser')->insert([
                'iStudentid' => $studentId,
                'sName' => $validated['sFirstName'],
                'iBisActive' => 1,
                'dtUpdatedDate' => now()
            ]);
    
            // 3️⃣ Auto insert into tblSocialCategoryOfStudents (EMPTY DEFAULT)
            DB::table('tblSocialCategoryOfStudents')->insert([
                'iStudentId' => $studentId,
                'dtUpdatedDate' => now(),
                'iBisActive' => 1
            ]);
    
            // 4️⃣ Auto insert into tblStudentQuotaDetails (EMPTY DEFAULT)
            DB::table('tblStudentQuotaDetails')->insert([
                'iStudentId' => $studentId,
                'dtUpdatedDate' => now(),
                'iBisActive' => 1
            ]);
    
            DB::commit();
    
            return response()->json([
                'result' => 1,
                'message' => 'Student & Default Related Records Created',
                'iStudentId' => $studentId
            ], 201);
    
        } catch (\Exception $e) {
            DB::rollBack();
    
            return response()->json([
                'result' => 0,
                'message' => 'Transaction Failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }


 
    //Add student basic student details  and also for App Register
    public function storeStudentsBasicDetails(Request $request)
    {
        try {
            $validated = $request->validate([
                'sFirstName' => 'required|string|max:50',
                'sMiddleName' => 'nullable|string|max:50',
                'sLastName' => 'required|string|max:50',
                'sGender' => 'nullable|string|max:20',
                'sCity' => 'nullable|string|max:50',
                'sStudentMobileNo' => [
                    'required',
                    'string',
                    'max:20',
                    Rule::unique('tblregistration', 'sStudentMobileNo')
                        ->where(function ($query) {
                            return $query->where('iBisActive', 1);
                        })
                ],
                'sParentMobileNo' => 'nullable|string|max:20',
                'sEmailId' => 'nullable|email|max:50',
                'sAppFlag' =>'nullable|integer',
                'iFiscalyearid' => 'nullable|integer',
                'sActiveStatus' => 'nullable|string',
                'iAppPackageId' => 'nullable|array',
                'iAppPackageId.*' => 'array',
            ]);
    
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'result' => 5,
                'message' => 'Validation Failed',
                'errors' => $e->errors()
            ], 422);
        }
    
        try {
            $studentId = DB::table('tblregistration')->insertGetId([
                'sFirstName' => $validated['sFirstName'],
                'sMiddleName' => $validated['sMiddleName'] ?? null,
                'sLastName' => $validated['sLastName'],
                'sGender' => $validated['sGender'] ?? null,
                'sCity' => $validated['sCity'] ?? null,
                'sStudentMobileNo' => $validated['sStudentMobileNo'],
                'sParentMobileNo' => $validated['sParentMobileNo'] ?? null,
                'sEmailId' => $validated['sEmailId'] ?? null,
                'sActiveStatus' => $validated['sActiveStatus'] ?? 'Active',
                'iFiscalyearid' => $validated['iFiscalyearid'] ?? 1,
                 'iAppPackageId' => isset($validated['iAppPackageId'])
                ? json_encode($validated['iAppPackageId'])
                : json_encode([]),
    
                'iBisActive' => 1,
                'dtUpdatedDate' => now(),
                 'sAppFlag' => $validated['sAppFlag'] ?? 1,
            ]);
    
            return response()->json([
                'result' => 1,
                'message' => 'Student Basic Details Saved Successfully',
                'iStudentId' => $studentId
            ], 201);
    
        } catch (\Exception $e) {
            return response()->json([
                'result' => 0,
                'message' => 'Failed to Save Student Details',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    //Update Basic Details of students while appointment
    public function updateStudentsBasicDetails(Request $request)
    {
        try {
            $validated = $request->validate([
                'iId' => 'required|integer|exists:tblregistration,iId',
    
                // basic details
                'sFirstName' => 'required|string|max:50',
                'sMiddleName' => 'nullable|string|max:50',
                'sLastName' => 'required|string|max:50',
                'sGender' => 'nullable|string|max:20',
                'sCity' => 'nullable|string|max:50',
                'sStudentMobileNo' => [
                    'required',
                    'string',
                    'max:20',
                   Rule::unique('tblregistration', 'sStudentMobileNo')
                ->where(function ($query) {
                    return $query->where('iBisActive', 1);
                })
                ->ignore($request->iId, 'iId')
    
                ],
                'sParentMobileNo' => 'nullable|string|max:20',
                'sEmailId' => 'nullable|email|max:50',
    
                // payment
                'iAmount' => 'nullable|integer|min:0',
                'iPartialAmount' => 'nullable|integer|min:0',
                'iConcession' => 'nullable|integer|min:0',
                'iTotalAmount' => 'nullable|integer|min:0',
    
                // group/status
                'sGroup' => 'nullable|string|max:10',
                'sActiveStatus' => 'nullable|string|max:20',
                'sFlag' => 'nullable|string|max:20',
    
                // packages
                'iCounsellingPackageId' => 'nullable|array',
                'iCounsellingPackageId.*' => 'array',
                'iAppPackageId' => 'nullable|array',
                'iAppPackageId.*' => 'array',
    
                // other details
                'sOtherDetails1' => 'nullable|array',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'result' => 5,
                'message' => 'Validation Failed',
                'errors' => $e->errors()
            ], 422);
        }
    
        try {
            $updated = DB::table('tblregistration')
                ->where('iId', $validated['iId'])
                ->update([
                    'sFirstName' => $validated['sFirstName'],
                    'sMiddleName' => $validated['sMiddleName'] ?? null,
                    'sLastName' => $validated['sLastName'],
                    'sGender' => $validated['sGender'] ?? null,
                    'sCity' => $validated['sCity'] ?? null,
                    'sStudentMobileNo' => $validated['sStudentMobileNo'],
                    'sParentMobileNo' => $validated['sParentMobileNo'] ?? null,
                    'sEmailId' => $validated['sEmailId'] ?? null,
    
                    'iAmount' => $validated['iAmount'] ?? DB::raw('iAmount'),
                    'iPartialAmount' => $validated['iPartialAmount'] ?? DB::raw('iPartialAmount'),
                    'iConcession' => $validated['iConcession'] ?? DB::raw('iConcession'),
                    'iTotalAmount' => $validated['iTotalAmount'] ?? DB::raw('iTotalAmount'),
    
                    'sGroup' => $validated['sGroup'] ?? DB::raw('sGroup'),
                    'sActiveStatus' => $validated['sActiveStatus'] ?? DB::raw('sActiveStatus'),
                    'sFlag' => $validated['sFlag'] ?? DB::raw('sFlag'),
    
                    'iCounsellingPackageId' => isset($validated['iCounsellingPackageId']) ? json_encode($validated['iCounsellingPackageId']) : json_encode([]),
                    'iAppPackageId' => isset($validated['iAppPackageId']) ? json_encode($validated['iAppPackageId']) : json_encode([]),
                    'sOtherDetails1' => isset($validated['sOtherDetails1']) ? json_encode($validated['sOtherDetails1']) : json_encode([]),
    
                    'dtUpdatedDate' => now(),
                ]);
    
            if ($updated) {
                return response()->json([
                    'result' => 1,
                    'message' => 'Student Details Updated Successfully'
                ], 200);
            }
    
            return response()->json([
                'result' => 0,
                'message' => 'No Changes Made'
            ], 200);
    
        } catch (\Exception $e) {
            return response()->json([
                'result' => 0,
                'message' => 'Failed to Update Student Details',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    
    //get completed todays booking and appointment list of students for account section
    public function getCompletedStudentsByDate(Request $request)
    {
        $request->validate([
            'dtDate' => 'required|date'
        ]);
    
        $date = $request->dtDate;
    
        $studentFields = [
            's.iId',
            's.iRegistrationId',
            's.sFirstName',
            's.sMiddleName',
            's.sLastName',
            's.sGender',
            's.sCity',
            's.sStudentMobileNo',
            's.sParentMobileNo',
            's.sEmailId',
            's.iAmount',
            's.iPartialAmount',
            's.iConcession',
            's.iTotalAmount',
            's.iCounsellingPackageId',
            's.iAppPackageId',
            's.sActiveStatus',
            's.sFlag',
            's.sParentDeviceId',
            's.sStudentDeviceId',
            's.sOTP',
            's.iBisActive',
            's.iUpdateby',
            's.sGroup',
        ];
    
        // 🔹 Payment subquery (total paid per student)
        $paymentSubquery = DB::table('tblPayment')
            ->select('iStudentid', DB::raw('SUM(fPaidAmount) as paid_amount'))
            ->groupBy('iStudentid');
    
        // 🔹 Completed Appointments
        $appointments = DB::table('tblAppointment as a')
            ->leftJoin('tblregistration as s', 'a.iStudentId', '=', 's.iId')
            ->leftJoinSub($paymentSubquery, 'p', function ($join) {
                $join->on('s.iId', '=', 'p.iStudentid');
            })
            ->where('a.iBisActive', 1)
            ->where('s.iBisActive', 1)
            ->where('a.sStatus', 'Completed')
            ->whereDate('a.dtDate', $date)
            ->select(array_merge(
                $studentFields,
                [
                    DB::raw('COALESCE(p.paid_amount, 0) as paid_amount'),
                    DB::raw("'Appointment' as source")
                ]
            ));
    
        // 🔹 Completed Slot Bookings
        $slotBookings = DB::table('tblSlotBooking as sb')
            ->leftJoin('tblregistration as s', 'sb.iStudentId', '=', 's.iId')
            ->leftJoinSub($paymentSubquery, 'p', function ($join) {
                $join->on('s.iId', '=', 'p.iStudentid');
            })
            ->where('sb.iBisActive', 1)
            ->where('s.iBisActive', 1)
            ->where('sb.sStatus', 'Completed')
            ->whereDate('sb.dtUpdatedDate', $date)
            ->select(array_merge(
                $studentFields,
                [
                    DB::raw('COALESCE(p.paid_amount, 0) as paid_amount'),
                    DB::raw("'SlotBooking' as source")
                ]
            ));
    
        // 🔹 Combine both
        $students = $appointments->union($slotBookings)->get();
    
        return response()->json([
            'result'  => 1,
            'message' => $students->count() ? 'Data Found' : 'No Data Found',
            'data'    => $students
        ]);
    }





    //get list of students who registred from APp for only 
    public function getAppRegisterdStudent(Request $request)
    {
        $fiscalYearId = $request->input('iFiscalYearId');

        if (!$fiscalYearId) {
            return response()->json([
                'result' => 0,
                'message' => 'Fiscal Year is required'
            ], 400);
        }
        // Subquery: payment totals per student
        $paymentSubquery = DB::table('tblPayment')
            ->select('iStudentid', DB::raw('SUM(fPaidAmount) as paid_amount'))
            ->where('iFiscalYearId', $fiscalYearId)
            ->groupBy('iStudentid');
    
        // Main query with category join (no groupBy issue)
        $registrations = DB::table('tblregistration as r')
            ->leftJoinSub($paymentSubquery, 'p', function ($join) {
                $join->on('r.iId', '=', 'p.iStudentid');
            })
            ->leftJoin('tblSocialCategoryOfStudents as c', 'r.iId', '=', 'c.iStudentid')
            ->select(
                'r.*',
                DB::raw('COALESCE(p.paid_amount, 0) as paid_amount'),
                'c.sCategory'
            )
            ->where('r.iBisActive', 1)
            ->where('r.iFiscalYearId', $fiscalYearId)
            ->where('r.sAppFlag', 0)
            ->get();
    
        if ($registrations->isEmpty()) {
            return response()->json([
                'result' => "6",
                'message' => 'No active registrations found',
            ], 404);
        }
    
        // Define prefix priority
        $prefixPriority = [
            'ME' => 0, 'EN' => 1, 'PH' => 2, 'AG' => 3,
            'AR' => 4, 'MB' => 5, 'MC' => 6, 'DS' => 7, 'ts' => 8,
        ];
    
        // Filter valid registrations
        $validRegistrations = $registrations->filter(function ($item) use ($prefixPriority) {
    if (!preg_match('/^([A-Za-z]+)(\d+)$/', $item->iRegistrationId, $matches)) {
        return false;
    }

    $rawPrefix = $matches[1];

    // ✅ remove trailing A (App flag)
        $prefix = rtrim($rawPrefix, 'A');
    
        return isset($prefixPriority[$prefix]);
    });

    
        // Filter unknown registrations
        $unknownRegistrations = $registrations->filter(function ($item) {
            return empty($item->iRegistrationId) || preg_match('/Unknown/', $item->iRegistrationId);
        });
    
        // Sort valid registrations
         $validRegistrations = $validRegistrations->sortBy(function ($item) use ($prefixPriority) {
            preg_match('/^([A-Za-z]+)(\d+)$/', $item->iRegistrationId, $matches);
        
            $rawPrefix = $matches[1] ?? 'Unknown';
        
            // ✅ remove trailing A
            $prefix = rtrim($rawPrefix, 'A');
        
            $number = (int) ($matches[2] ?? 0);
            $priority = $prefixPriority[$prefix] ?? 99;
        
            return [$priority, $number];
        });

    
        // Merge final list
        $registrations = $validRegistrations->merge($unknownRegistrations);
    
        // Final response
        return response()->json([
            'result' => "1",
            'message' => 'Active registrations retrieved successfully',
            'data' => $registrations,
        ], 200);
    }
    
    
    
    //get filtered registertion for App
    public function getFilteredRegistrationsForApp(Request $request)
    {
        $program = $request->input('program');
        $status = $request->input('status');
        $paymentStatus = $request->input('payment'); // 'paid', 'partial', 'pending'
        $fiscalYearId = $request->input('iFiscalYearId');
    
        if (!$fiscalYearId) {
            return response()->json([
                'result' => 0,
                'message' => 'Fiscal Year is required'
            ], 400);
        }
    
        // Subquery: calculate total paid per student
        $payments = DB::table('tblPayment')
            ->select('iStudentid', DB::raw('SUM(fPaidAmount) as paid_amount'))
            ->where('iFiscalYearId', $fiscalYearId)
            ->groupBy('iStudentid');
    
        // Main query: join registrations with payments
        $query = DB::table('tblregistration as r')
            ->leftJoinSub($payments, 'p', function ($join) {
                $join->on('r.iId', '=', 'p.iStudentid');
            })
            ->select(
                'r.*',
                DB::raw('COALESCE(p.paid_amount, 0) as paid_amount')
            )
            ->where('r.iBisActive', 1)        // only active students
            ->where('r.iFiscalYearId', $fiscalYearId)  // fiscal year filter
            ->where('r.sAppFlag', 0);       // only App-registered student
    
        // Optional: filter by program
        if ($program) {
            $query->where(function ($q) use ($program) {
                $q->whereRaw("JSON_CONTAINS(r.iCounsellingPackageId, JSON_OBJECT('sProgram', ?))", [$program])
                  ->orWhereRaw("JSON_CONTAINS(r.iAppPackageId, JSON_OBJECT('sProgram', ?))", [$program]);
            });
        }
    
        // Optional: filter by active status
        if ($status) {
            $query->where('r.sActiveStatus', $status);
        }
    
        // Optional: filter by payment status
        if ($paymentStatus === 'paid') {
            $query->whereRaw('COALESCE(p.paid_amount, 0) = r.iTotalAmount');
        } elseif ($paymentStatus === 'partial') {
            $query->whereRaw('COALESCE(p.paid_amount, 0) > 0 AND COALESCE(p.paid_amount, 0) < r.iTotalAmount');
        } elseif ($paymentStatus === 'pending') {
            $query->whereRaw('COALESCE(p.paid_amount, 0) = 0');
        }
    
        $results = $query->get();
    
        // Add dynamic payment status label
        $results->transform(function ($item) {
            if ($item->paid_amount == 0) {
                $item->payment_status = 'Pending';
            } elseif ($item->paid_amount < $item->iTotalAmount) {
                $item->payment_status = 'Partial';
            } else {
                $item->payment_status = 'Paid';
            }
            return $item;
        });
    
        return response()->json([
            'result' => 1,
            'message' => 'Filtered records fetched successfully',
            'data' => $results
        ]);
    }
    
    
    
    
    //get studentlist for verfication who registered from App
    public function getAppStudentListForVerification(Request $request)
    {
        try {
            $validated = $request->validate([
                'iFiscalYearId' => 'required|integer',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'result' => 5,
                'message' => 'Validation Failed',
                'errors' => $e->errors()
            ], 422);
        }
    
        try {
            $students = DB::table('tblregistration')
                ->where('sAppFlag', 0)
                ->where('sActiveStatus', 'Inactive')
                ->where('iFiscalyearid', $validated['iFiscalYearId'])
                ->where('iBisActive', 1)
                ->select(
                    'iId',
                    'sFirstName',
                    'sMiddleName',
                    'sLastName',
                    'sStudentMobileNo',
                    'sParentMobileNo',
                    'sEmailId',
                    'sActiveStatus',
                    'sAppFlag',
                    'iFiscalyearid',
                    'iAppPackageId',
                    'dtUpdatedDate'
                )
                ->orderBy('dtUpdatedDate', 'desc')
                ->get();
    
            return response()->json([
                'result' => 1,
                'message' => 'Student Verification List Fetched Successfully',
                'data' => $students
            ], 200);
    
        } catch (\Exception $e) {
            return response()->json([
                'result' => 0,
                'message' => 'Failed to Fetch Student List',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    public function verifyStudentAfterPayment(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'iId' => 'required|integer|exists:tblregistration,iId',
        ]);
    
        if ($validator->fails()) {
            return response()->json([
                'result' => 5,
                'message' => 'Validation Failed',
                'errors' => $validator->errors()
            ], 422);
        }
    
        try {
            DB::beginTransaction();
    
            $student = DB::table('tblregistration')
                ->where('iId', $request->iId)
                ->where('iBisActive', 1)
                ->first();
    
            if (!$student) {
                return response()->json([
                    'result' => 0,
                    'message' => 'Student not found'
                ], 404);
            }
    
            if ($student->sActiveStatus === 'Active') {
                return response()->json([
                    'result' => 2,
                    'message' => 'Student already verified'
                ], 200);
            }
    
            /* ===============================
               🔢 REGISTRATION ID GENERATION
               (FROM iAppPackageId)
            ================================*/
    
            if (empty($student->iRegistrationId)) {
    
                // ✅ USE APP PACKAGE
                $packages = json_decode($student->iAppPackageId, true);
                $firstPackage = $packages[0] ?? null;
    
                if (!$firstPackage || empty($firstPackage['sProgram'])) {
                    return response()->json([
                        'result' => 6,
                        'message' => 'Program not found for student'
                    ], 422);
                }
    
                $programPrefixes = [
                    'Medical' => 'ME',
                    'Engineering' => 'EN',
                    'Pharmacy' => 'PH',
                    'Agriculture' => 'AG',
                    'Architecture' => 'AR',
                    'MBA' => 'MB',
                    'MCA' => 'MC',
                    'DSY Engineering' => 'DS',
                ];
    
                $programName = $firstPackage['sProgram'];
                $programPrefix = $programPrefixes[$programName] ?? null;
    
                if (!$programPrefix) {
                    return response()->json([
                        'result' => 6,
                        'message' => 'Invalid program for ID generation'
                    ], 422);
                }
    
                $appPrefix = 'A'; // ✅ APP IDENTIFIER
                $yearPrefix = substr(date('Y'), -2);
    
                $lastStudent = DB::table('tblregistration')
                    ->where('iRegistrationId', 'LIKE', "{$programPrefix}{$appPrefix}{$yearPrefix}%")
                    ->orderBy('iRegistrationId', 'desc')
                    ->first();
    
                $lastNumber = $lastStudent
                    ? intval(substr(
                        $lastStudent->iRegistrationId,
                        strlen($appPrefix . $programPrefix . $yearPrefix)
                    ))
                    : 0;
    
                $registrationId = $appPrefix
                    . $programPrefix
                    . $yearPrefix
                    . str_pad($lastNumber + 1, 4, '0', STR_PAD_LEFT);
    
            } else {
                $registrationId = $student->iRegistrationId;
            }
    
            /* ===============================
               🔑 ACTIVATE + PASSWORD
            ================================*/
    
            DB::table('tblregistration')
                ->where('iId', $student->iId)
                ->update([
                    'iRegistrationId' => $registrationId,
                    'sActiveStatus' => 'Active',
                    'dtUpdatedDate' => now()
                ]);
    
            /* ===============================
               📲 WHATSAPP MESSAGE
            ================================*/
    
            // try {
            //     Http::post('https://mysoftway.com/whatsapp/webhook.php', [
            //         'template' => 'account_creation_confirmation',
            //         'to' => '91' . $student->sStudentMobileNo,
            //         'name' => $student->sFirstName,
            //         'StudentMobile' => $student->sStudentMobileNo,
            //         'ParentMobile' => $student->sParentMobileNo,
            //         'password' => $generatedPassword,
            //         'registrationId' => $registrationId
            //     ]);
            // } catch (\Exception $e) {
            //     Log::error('WhatsApp send failed: ' . $e->getMessage());
            // }
    
            DB::commit();
    
            return response()->json([
                'result' => 1,
                'message' => 'Student verified successfully',
                'data' => [
                    'iStudentId' => $student->iId,
                    'iRegistrationId' => $registrationId
                ]
            ], 200);
    
        } catch (\Exception $e) {
            DB::rollBack();
    
            Log::error('verifyStudentAfterPayment error', [
                'error' => $e->getMessage()
            ]);
    
            return response()->json([
                'result' => 0,
                'message' => 'Failed to verify student',
                'error' => $e->getMessage()
            ], 500);
        }
    }



}







