<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;


class CircularController extends Controller
{
    
    // Add circular
    public function addCircular(Request $request)
    {
        try {
            $validatedData = $request->validate([
                'sProgram' => 'required|string|max:100',
                'dtDate' => 'required|date',
                'sCircularName' => 'required|string|max:100',
                'sDocumentUrl' => 'required|string|max:255',
                'sOtherDetails1' => 'nullable|integer',
                'sOtherDetails2' => 'nullable|integer',
                'sOtherDetails3' => 'nullable|integer',
                'iFiscalYearId' => 'required|integer|exists:tblFiscalYear,iId'
            ]);
    
            $circularId = DB::table('tblNotification')->insertGetId([
                'sProgram' => $validatedData['sProgram'],
                'dtDate' => $validatedData['dtDate'],
                'sCircularName' => $validatedData['sCircularName'],
                'sDocumentUrl' => $validatedData['sDocumentUrl'],
                'dtUpdatedDate' => now(),
                'iBisActive' => 1,
                'sOtherDetails1' => $validatedData['sOtherDetails1'] ?? null,
                'sOtherDetails2' => $validatedData['sOtherDetails2'] ?? null,
                'sOtherDetails3' => $validatedData['sOtherDetails3'] ?? null,
                'iFiscalYearId' => $validatedData['iFiscalYearId'],
            ]);
    
            return response()->json([
                'result' =>"2",
                'message' => 'Circular added successfully',
                'id' => $circularId
            ], 201);
    
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'result' => "0",
                'message' => 'Validation error',
                'errors' => $e->errors()
            ], 422);
    
        } catch (\Exception $e) {
            return response()->json([
                'result' => "0",
                'message' => 'Error occurred while adding circular',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    
    // udate circular 
    public function updateCircular(Request $request)
    {
    try {
        $validatedData = $request->validate([
            'iId' => 'required|numeric',
            'sProgram' => 'required|string|max:100',
            'dtDate' => 'required|date',
            'sCircularName' => 'required|string|max:100',
            'sDocumentUrl' => 'required|string|max:255',
            'sOtherDetails1' => 'nullable|integer',
            'sOtherDetails2' => 'nullable|integer',
            'sOtherDetails3' => 'nullable|integer',
        ]);

        $id = $validatedData['iId'];

        $circular = DB::table('tblNotification')
            ->where('iId', $id)
            ->where('iBisActive', 1)
            ->first();

        if (!$circular) {
            return response()->json([
                'result' => 0,
                'message' => 'Circular not found or already deleted'
            ], 404);
        }

        DB::table('tblNotification')
            ->where('iId', $id)
            ->update([
                'sProgram' => $validatedData['sProgram'],
                'dtDate' => $validatedData['dtDate'],
                'sCircularName' => $validatedData['sCircularName'],
                'sDocumentUrl' => $validatedData['sDocumentUrl'],
                'dtUpdatedDate' => now(),
                'sOtherDetails1' => $validatedData['sOtherDetails1'] ?? null,
                'sOtherDetails2' => $validatedData['sOtherDetails2'] ?? null,
                'sOtherDetails3' => $validatedData['sOtherDetails3'] ?? null,
            ]);

        return response()->json([
            'result' => "3",
            'message' => 'Circular updated successfully',
            'id' => $id
        ], 200);

    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json([
            'result' => "0",
            'message' => 'Validation error',
            'errors' => $e->errors()
        ], 422);

    } catch (\Exception $e) {
        return response()->json([
            'result' => "0",
            'message' => 'Error occurred while updating circular',
            'error' => $e->getMessage()
        ], 500);
    }
}


    // get circular
    public function getCircular($id)
    {
        try {
            $circular = DB::table('tblNotification')->where('iId', $id)->where('iBisActive', 1)->first();
    
            if (!$circular) {
                return response()->json([
                    'result' => "0",
                    'message' => 'Circular not found'
                ], 404);
            }
    
            return response()->json([
                'result' =>"1",
                'message' => 'Circular fetched successfully',
                'data' => $circular
            ], 200);
    
        } catch (\Exception $e) {
            return response()->json([
                'result' => "0",
                'message' => 'Error occurred while fetching circular',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    //delete circular
    public function deleteCircular($id)
    {
        try {
            $circular = DB::table('tblNotification')->where('iId', $id)->where('iBisActive', 1)->first();
    
            if (!$circular) {
                return response()->json([
                    'result' => "0",
                    'message' => 'Circular not found or already deleted'
                ], 404);
            }
    
            DB::table('tblNotification')->where('iId', $id)->update(['iBisActive' => 0, 'dtUpdatedDate' => now()]);
    
            return response()->json([
                'result' => "4",
                'message' => 'Circular deleted successfully (soft delete)'
            ], 200);
    
        } catch (\Exception $e) {
            return response()->json([
                'result' =>"0",
                'message' => 'Error occurred while deleting circular',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    //get all circulars    
    public function getAllCirculars(Request $request)
    {
        $fiscalYearId = $request->input('iFiscalYearId');

        if (!$fiscalYearId) {
            return response()->json([
                'result' => 0,
                'message' => 'Fiscal Year is required'
            ], 400);
        }

        try {
            $circulars = DB::table('tblNotification')
            ->where('iBisActive', 1)
            ->where('iFiscalYearId', $fiscalYearId)
            ->get();
    
            if ($circulars->isEmpty()) {
                return response()->json([
                    'result' => 0,
                    'message' => 'No circulars found'
                ], 404);
            }
    
            return response()->json([
                'result' => "1",
                'message' => 'Circulars fetched successfully',
                'data' => $circulars
            ], 200);
    
        } catch (\Exception $e) {
            return response()->json([
                'result' => "0",
                'message' => 'Error occurred while fetching circulars',
                'error' => $e->getMessage()
            ], 500);
        }
    }


    //get circular by program
    public function getCircularsByProgram(Request $request)
    {
    try {
        $validatedData = $request->validate([
            'sProgram' => 'required|string|max:100',
        ]);

        $circulars = DB::table('tblNotification')
            ->where('sProgram', $validatedData['sProgram'])
            ->where('iBisActive', 1) // Only active notifications
            ->orderBy('dtDate', 'desc') // Latest first (optional)
            ->get();

        return response()->json([
            'result' => "2",
            'message' => 'Circulars fetched successfully',
            'data' => $circulars
        ], 200);

    } catch (\Illuminate\Validation\ValidationException $e) {
        return response()->json([
            'result' => "0",
            'message' => 'Validation error',
            'errors' => $e->errors()
        ], 422);

    } catch (\Exception $e) {
        return response()->json([
            'result' => "0",
            'message' => 'Error occurred while fetching circulars',
            'error' => $e->getMessage()
        ], 500);
    }
}

}